//
//  UILabelX.swift
//  DesignableXTesting
//
//  Created by Tushar Premal on 1/28/17.
//  Copyright © 2017 Tushar Premal. All rights reserved.
//

import UIKit

@IBDesignable
class UILabelX: UILabel {
    
    @IBInspectable public var cornerRadius: CGFloat = 0 {
        didSet {
            if cornerRadius > 0{
                self.clipsToBounds = true
            }
            self.layer.cornerRadius = cornerRadius
        }
    }
    
    @IBInspectable var borderWidth: CGFloat = 0.0 {
        didSet {
            self.layer.borderWidth = borderWidth
        }
    }
    
    @IBInspectable var borderColor: UIColor = UIColor.clear {
        didSet {
            self.layer.borderColor = borderColor.cgColor
        }
    }
    
    @IBInspectable var rotationAngle: CGFloat = 0 {
        didSet {
            self.transform = CGAffineTransform(rotationAngle: rotationAngle * .pi / 180)
        }
    }
    
    // MARK: - Shadow Text Properties
    
    @IBInspectable public var shadowOpacity: CGFloat = 0 {
        didSet {
            layer.shadowOpacity = Float(shadowOpacity)
        }
    }
    
    @IBInspectable public var shadowColorLayer: UIColor = UIColor.clear {
        didSet {
            layer.shadowColor = shadowColorLayer.cgColor
        }
    }
    
    @IBInspectable public var shadowRadius: CGFloat = 0 {
        didSet {
            layer.shadowRadius = shadowRadius
        }
    }
    
    @IBInspectable public var shadowOffsetY: CGFloat = 0 {
        didSet {
            layer.shadowOffset.height = shadowOffsetY
        }
    }
    
    @IBInspectable open var characterSpacing:CGFloat = 1 {
        didSet {
            let attributedString = NSMutableAttributedString(string: self.text!)
            attributedString.addAttribute(NSAttributedString.Key.kern, value: self.characterSpacing, range: NSRange(location: 0, length: attributedString.length))
            self.attributedText = attributedString
        }
        
    }

}

class MarqueeLabel: UIView {
    
    private let textLabel1 = UILabel()
    private let textLabel2 = UILabel()
    private var animationDuration: Double = 0
    private var isAnimating = false
    
    var text: String? {
        didSet {
            textLabel1.text = text
            textLabel2.text = text
            layoutIfNeeded()
            startMarquee()
        }
    }
    
    var textColor: UIColor {
        get { textLabel1.textColor }
        set {
            textLabel1.textColor = newValue
            textLabel2.textColor = newValue
        }
    }
    
    var font: UIFont {
        get { textLabel1.font }
        set {
            textLabel1.font = newValue
            textLabel2.font = newValue
        }
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupLabels()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setupLabels()
    }
    
    private func setupLabels() {
        clipsToBounds = true
        
        [textLabel1, textLabel2].forEach {
            $0.textAlignment = .left
            $0.lineBreakMode = .byClipping
            addSubview($0)
        }
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        startMarquee()
    }
    
    private func startMarquee() {
        guard let text = text, !text.isEmpty, !isAnimating else { return }

        let textSize = textLabel1.intrinsicContentSize
        
        // If the label fits, no animation needed
        if textSize.width <= bounds.width {
            textLabel1.frame = bounds
            textLabel2.isHidden = true
            return
        }
        
        // Place two labels side by side
        textLabel1.frame = CGRect(x: 0, y: 0, width: textSize.width, height: bounds.height)
        textLabel2.frame = CGRect(x: textSize.width + 100, y: 0, width: textSize.width, height: bounds.height)
        textLabel2.isHidden = false
        
        let totalWidth = textSize.width
        
        animationDuration = Double(totalWidth) / 25.0  // 30 pts/sec

        // Animate both labels to the left continuously
        isAnimating = true
        animateScroll()
    }
    
    private func animateScroll() {
        let textSize = textLabel1.intrinsicContentSize
        textLabel1.frame.origin.x = 0
        textLabel2.frame.origin.x = textSize.width + 100
        
        UIView.animate(withDuration: animationDuration, delay: 0, options: [.curveLinear, .repeat]) {
            self.textLabel1.frame.origin.x = -(textSize.width+100)
            self.textLabel2.frame.origin.x = 0
        }
    }
}

class MarqueeManager {
    static let shared = MarqueeManager()
    
    private var marqueeLabel: MarqueeLabel?
    
    /// Show a red marquee at the bottom of every screen (iOS 15+).
    func show(text: String) {
        // 1. Find the key window via connectedScenes (iOS 15+)
        guard let windowScene = UIApplication.shared.connectedScenes
                .compactMap({ $0 as? UIWindowScene })
                .first(where: { $0.activationState == .foregroundActive }),
              let keyWindow = windowScene.windows.first(where: { $0.isKeyWindow })
        else {
            return
        }

        // 2. Remove any existing marquee
        marqueeLabel?.removeFromSuperview()
        marqueeLabel = nil

        // 3. Compute frame: bottom of screen, above safeAreaInsets
        let height: CGFloat = 40
        let safeBottom: CGFloat = keyWindow.safeAreaInsets.bottom
        let yPosition = keyWindow.bounds.height - height - safeBottom
        let frame = CGRect(x: (keyWindow.bounds.width-600)/2, y: yPosition, width: 600, height: height)

        // 4. Create and configure MarqueeLabel
        let label = MarqueeLabel(frame: frame)
        label.backgroundColor = .red
        label.textColor = .white
        label.font = UIFont.boldSystemFont(ofSize: 15)
        label.text = text

        // 5. Add to window and bring to front
        keyWindow.addSubview(label)
        keyWindow.bringSubviewToFront(label)
        
        marqueeLabel = label
    }
    
    /// Hide the marquee if it’s currently shown
    func hide() {
        marqueeLabel?.removeFromSuperview()
        marqueeLabel = nil
    }
}
