//
//  SQLiteManager.swift
//  EPOS
//
//  Created by Apple on 22/06/21.
//

import Foundation
import SQLite3
import CoreData

class SQLiteManage {
    
    static let SQLITE_STATIC = unsafeBitCast(0, to: sqlite3_destructor_type.self)
    static let SQLITE_TRANSIENT = unsafeBitCast(-1, to: sqlite3_destructor_type.self)
    
    //MARK: Database Operations
    static func getDBPath()->URL?{
        if let destPath = NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask,  true).first {
            let fileName = "EPOS.sqlite"
            return URL(fileURLWithPath: destPath).appendingPathComponent(fileName)
        }
        return nil
    }
    
    static func copyDatabaseToDirectory()->String{
        if let fullDestPath = getDBPath() {
            let fullDestPathString = fullDestPath.path
            let fileManage = FileManager.default
            if !fileManage.fileExists(atPath: fullDestPathString) {
                
                let bundlePath = Bundle.main.path(forResource: "EPOS", ofType: ".sqlite")!
                do {
                    try fileManage.copyItem(atPath: bundlePath, toPath: fullDestPathString)
                } catch let error {
                    print(error.localizedDescription)
                }
            }
            return fullDestPathString
        }
        return ""
    }
    
    static func printDBPath(){
        if let fullDestPath = getDBPath() {
            let fullDestPathString = fullDestPath.path
            if FileManager.default.fileExists(atPath: fullDestPathString){
//                migrateToNewVersion()
            }
            print("db path==>", fullDestPathString)
        }
    }
    
    static func checkColumnExist(table:String,column:String)->Bool{
        let pragmaProduct = SelectQuery("PRAGMA table_info (\(table))")
        return pragmaProduct.contains { $0["name"] == column }
    }
    
    static func fetchCountOfTable(whereString:String,tableName:String)->Int{
        let query = "select count(*) from \(tableName) \(whereString)"
        let DBpath:String = copyDatabaseToDirectory()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=query.cString(using: String.Encoding.utf8)
        
        var count = 0
        var db: OpaquePointer? = nil
        if (sqlite3_open(DBpath, &db)==SQLITE_OK){
            if sqlite3_prepare_v2(db, strExec!, -1, &stmt, nil) == SQLITE_OK{
                while(sqlite3_step(stmt) == SQLITE_ROW){
                    count = Int(sqlite3_column_int(stmt, 0))
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        return count
    }
    
    static func fetchSumOfTable(feild:String, whereString:String,tableName:String)->Double{
        let query = "select sum(\(feild)) from \(tableName) \(whereString)"
        let DBpath:String = copyDatabaseToDirectory()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=query.cString(using: String.Encoding.utf8)
        
        var count:Double = 0
        var db: OpaquePointer? = nil
        if (sqlite3_open(DBpath, &db)==SQLITE_OK){
            if sqlite3_prepare_v2(db, strExec!, -1, &stmt, nil) == SQLITE_OK{
                while(sqlite3_step(stmt) == SQLITE_ROW){
                    count = sqlite3_column_double(stmt, 0)
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        return count
    }
    
    static func ExecuteQuery(_ str:String,_ db:OpaquePointer? = nil) -> Bool{
        var result:Bool=false
        let DBpath:String = copyDatabaseToDirectory()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if db != nil{
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                if (sqlite3_step(stmt) == SQLITE_DONE){
                    result = true
                }
            }
            sqlite3_finalize(stmt)
        }else{
            var db: OpaquePointer? = nil
            if (sqlite3_open(DBpath, &db)==SQLITE_OK){
                if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                    if (sqlite3_step(stmt) == SQLITE_DONE){
                        result = true
                    }
                }
                sqlite3_finalize(stmt)
            }
            sqlite3_close(db)
        }
        return result
    }
    
    static func SelectQuery(_ str:String) -> Array<Dictionary<String,String>>{
        var result:Array<Dictionary<String,String>>=[]
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    var i:Int32 = 0
                    let icount = sqlite3_column_count(stmt)
                    
                    var dict = Dictionary<String, String>()
                    
                    while i < icount{
                        if let strF=sqlite3_column_name(stmt, i){
                            let rFiled = String(cString: strF)
                            
                            if let strV = sqlite3_column_text(stmt, i){
                                dict[rFiled] = String(cString: strV)
                            }else{
                                dict[rFiled] = ""
                            }
                        }
                        i += 1
                    }
                    result.append(dict)
                }
                sqlite3_finalize(stmt)
            }
            
        }
        
        sqlite3_close(db)
        return result
    }
    
    //MARK: Tables
    static func FetchTables(_ str:String) -> [TableModel]{
        var result = [TableModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = TableModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.number = String(cString: sqlite3_column_text(stmt, 1))
                    mod.floorId = Int(sqlite3_column_int64(stmt, 2))
                    mod.lastOrderId = Int(sqlite3_column_int64(stmt, 3))
                    mod.lastOrderTime = String(cString: sqlite3_column_text(stmt, 4))
                    mod.lastOrderTotal = sqlite3_column_double(stmt, 5)
                    mod.offline = Int(sqlite3_column_int64(stmt, 7))
                    mod.sequence = Int(sqlite3_column_int64(stmt, 8))
                    mod.tableStatusId = Int(sqlite3_column_int64(stmt, 9))
                    mod.locked = Int(sqlite3_column_int64(stmt, 12)) == 1
                    mod.mergeTableId = Int(sqlite3_column_int64(stmt, 13))
                    mod.updaterId = Int(sqlite3_column_int64(stmt, 14))
                    mod.tableStatus = CoreDataHelper.shared.fetchTableStatusData(id: mod.tableStatusId).first
                    
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        
        return result
    }
    
    static func fetchTableData(floorId:Int, statusId:Int? = nil, noDisabled:Bool = true)->[TableModel]{
        if statusId != nil{
            if noDisabled{
                return FetchTables("SELECT * FROM Tables WHERE floor_id = \(floorId) AND disabled = 0 AND table_status_id = \(statusId!) ORDER BY sequence")
            }else{
                return FetchTables("SELECT * FROM Tables WHERE floor_id = \(floorId) AND table_status_id = \(statusId!) ORDER BY sequence")
            }
        }else{
            if noDisabled{
                return FetchTables("SELECT * FROM Tables WHERE floor_id = \(floorId) AND disabled = 0 ORDER BY sequence")
            }else{
                return FetchTables("SELECT * FROM Tables WHERE floor_id = \(floorId) ORDER BY sequence")
            }
        }
    }
    
    static func fetchTablesWithMergeId(floorId:Int, id:Int)->[TableModel]{
        return FetchTables("SELECT * FROM Tables WHERE floor_id = \(floorId) AND (table_status_id = 1 OR merge_table_id == \(id)) ORDER BY sequence")
    }
    
    static func fetchTableDataFromId(id:Int, statusId:Int? = nil)->TableModel?{
        return FetchTables("SELECT * FROM Tables WHERE id = \(id) LIMIT 1").first
    }
    
    static func fetchTableDataFromMergeId(id:Int)->TableModel?{
        return FetchTables("SELECT * FROM Tables WHERE merge_table_id = \(id) LIMIT 1").first
    }
    
    static func updateTableStatus(tableId:Int, status:String? = nil, statusId:Int? = nil, lastOrderId:Int? = nil, lastOrderTotal:Double? = nil, lastOrderTime:String? = nil, offline:Int? = nil, locked: Int? = nil, mergeTableId:Int? = nil){
        var strValues = ""
        if lastOrderId != nil{
            strValues = "last_order_id = \(lastOrderId!)"
        }
        if lastOrderTotal != nil{
            if strValues == ""{
                strValues = "last_order_total = \(lastOrderTotal!)"
            }else{
                strValues = "\(strValues), last_order_total = \(lastOrderTotal!)"
            }
        }
        if lastOrderTime != nil{
            if strValues == ""{
                strValues = "last_order_time = '\(lastOrderTime!)'"
            }else{
                strValues = "\(strValues), last_order_time = '\(lastOrderTime!)'"
            }
        }
        if statusId != nil{
            if strValues == ""{
                strValues = "table_status_id = \(statusId!)"
            }else{
                strValues = "\(strValues), table_status_id = \(statusId!)"
            }
        }
        if status != nil{
            if let tableStatus = CoreDataHelper.shared.fetchTableStatusData(name: status!).first{
                if strValues == ""{
                    strValues = "table_status_id = \(tableStatus.id ?? 0)"
                }else{
                    strValues = "\(strValues), table_status_id = \(tableStatus.id ?? 0)"
                }
            }else{
                AppCommonMethods.showToastAlert(message: "Table status not found")
                return
            }
        }
        if locked != nil{
            if strValues == ""{
                strValues = "locked = \(locked!)"
            }else{
                strValues = "\(strValues), locked = \(locked!)"
            }
            strValues = "\(strValues), updater_id = \(AppConstants.userData?.id ?? 0)"
        }
        if offline != nil{
            if strValues == ""{
                strValues = "offline = \(offline!)"
            }else{
                strValues = "\(strValues), offline = \(offline!)"
            }
        }
        if mergeTableId != nil{
            if strValues == ""{
                strValues = "merge_table_id = \(mergeTableId!)"
            }else{
                strValues = "\(strValues), merge_table_id = \(mergeTableId!)"
            }
        }
        if strValues != ""{
            let strQuery = "UPDATE Tables SET \(strValues) WHERE id = \(tableId)"
            _ = ExecuteQuery(strQuery)
        }
    }
    
    static func insertTable(arrData:[TableModel], floorId:Int? = nil){
        var arrTables = [TableModel]()
        var arr = [TableModel]()
        if floorId != nil{
            arr = fetchTableData(floorId: arrData.first?.floorId ?? 0, noDisabled: false)
        }else{
            arr = FetchTables("SELECT * FROM Tables ORDER BY sequence")
        }
        if arrData.first?.floorId ?? 0 != 0{
            arrTables = arr
            for object in arr{
                let isContain = arrData.contains { (model) -> Bool in
                    return model.id ?? 0 == object.id ?? 0
                }
                if !isContain{
                    _ = ExecuteQuery("DELETE FROM Tables WHERE id = \(object.id ?? 0)")
                }
            }
        }
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        let update = "UPDATE Tables SET number = ?, floor_id = ?, last_order_time = ?, last_order_total = ?, sequence = ?, table_status_id = ?, disabled = ?, last_order_id = ?, locked = ?, merge_table_id = ?, updater_id = ? WHERE id = ?"
        let updateSequence = "UPDATE Tables SET sequence = ?, disabled = ?, locked = ?, table_status_id = ?, merge_table_id = ?, updater_id = ? WHERE id = ?"
        let insert = "INSERT INTO Tables (id, number, floor_id, last_order_time, last_order_total, offline, sequence, table_status_id, disabled, last_order_id, locked, merge_table_id, updater_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK){
            for model in arrData{
                let obj = arrTables.filter { (object) -> Bool in
                    return model.id ?? 0 == object.id ?? 0
                }
                var statement: OpaquePointer!
                if obj.count > 0{
                    if obj[0].offline == 1{
                        if sqlite3_prepare_v2(database, updateSequence, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 7, sqlite3_int64(model.id ?? 0))
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(model.sequence))
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(model.disabled ? 1 : 0))
                            sqlite3_bind_int64(statement, 3, sqlite3_int64(model.locked ? 1 : 0))
                            sqlite3_bind_int64(statement, 4, sqlite3_int64(model.tableStatusId ?? 0))
                            sqlite3_bind_int64(statement, 5, sqlite3_int64(model.mergeTableId))
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(model.updaterId))
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not update table sequence, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }else{
                        if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(model.id ?? 0))
                            sqlite3_bind_text(statement, 1, ((model.number ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(model.floorId ?? 0))
                            sqlite3_bind_text(statement, 3, ((model.lastOrderTime ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 4, (model.lastOrderTotal ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 5, sqlite3_int64(model.sequence))
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(model.tableStatusId ?? 0))
                            sqlite3_bind_int64(statement, 7, sqlite3_int64(model.disabled ? 1: 0))
                            sqlite3_bind_int64(statement, 8, sqlite3_int64(model.lastOrderId ?? 0))
                            sqlite3_bind_int64(statement, 9, sqlite3_int64(model.locked ? 1: 0))
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(model.mergeTableId))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(model.updaterId))
                            
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not update table, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }else{
                    if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(model.id ?? 0))
                        sqlite3_bind_text(statement, 2, ((model.number ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(model.floorId ?? 0))
                        sqlite3_bind_text(statement, 4, ((model.lastOrderTime ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 5, (model.lastOrderTotal ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(0))
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(model.sequence))
                        sqlite3_bind_int64(statement, 8, sqlite3_int64(model.tableStatusId ?? 0))
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(model.disabled ? 1: 0))
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(model.lastOrderId ?? 0))
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(model.locked ? 1: 0))
                        sqlite3_bind_int64(statement, 12, sqlite3_int64(model.mergeTableId))
                        sqlite3_bind_int64(statement, 13, sqlite3_int64(model.updaterId))
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not insert table, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
            }
        }
        sqlite3_close(database)
    }
    
    static func updateTable(model:TableModel){
        if let table = self.fetchTableDataFromId(id: model.id ?? 0){
            let dbPath = copyDatabaseToDirectory()
            let update = "UPDATE Tables SET number = ?, floor_id = ?, last_order_time = ?, last_order_total = ?, offline = ?, sequence = ?, table_status_id = ?, disabled = ?, last_order_id = ?, locked = ?, merge_table_id = ?, updater_id = ? WHERE id = ?"
            let updateSequence = "UPDATE Tables SET sequence = ?, disabled = ?, locked = ?, table_status_id = ?, merge_table_id = ?, updater_id = ? WHERE id = ?"
            var database: OpaquePointer? = nil
            if (sqlite3_open(dbPath, &database)==SQLITE_OK){
                var statement: OpaquePointer!
                if table.offline == 1{
                    if sqlite3_prepare_v2(database, updateSequence, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(model.id ?? 0))
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(model.sequence))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(model.disabled ? 1: 0))
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(model.locked ? 1: 0))
                        sqlite3_bind_int64(statement, 4, sqlite3_int64(model.tableStatusId ?? 0))
                        sqlite3_bind_int64(statement, 5, sqlite3_int64(model.mergeTableId))
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(model.updaterId))
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not update table sequence, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }else{
                    if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 13, sqlite3_int64(model.id ?? 0))
                        sqlite3_bind_text(statement, 1, ((model.number ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(model.floorId ?? 0))
                        sqlite3_bind_text(statement, 3, ((model.lastOrderTime ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 4, (model.lastOrderTotal ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 5, sqlite3_int64(0))
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(model.sequence))
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(model.tableStatusId ?? 0))
                        sqlite3_bind_int64(statement, 8, sqlite3_int64(model.disabled ? 1: 0))
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(model.lastOrderId ?? 0))
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(model.locked ? 1: 0))
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(model.mergeTableId))
                        sqlite3_bind_int64(statement, 12, sqlite3_int64(model.updaterId))
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not update table, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
            }
            sqlite3_close(database)
        }
    }

    static func updateTableSocket(model:TableModel){
        let dbPath = copyDatabaseToDirectory()
        let update = "UPDATE Tables SET last_order_time = ?, last_order_total = ?, offline = ?, table_status_id = ?, last_order_id = ?, locked = ?, merge_table_id = ?, updater_id = ? WHERE id = ?"
        var database: OpaquePointer? = nil
        if (sqlite3_open(dbPath, &database)==SQLITE_OK){
            var statement: OpaquePointer!
            if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                sqlite3_bind_int64(statement, 9, sqlite3_int64(model.id ?? 0))
                sqlite3_bind_text(statement, 1, ((model.lastOrderTime ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                sqlite3_bind_double(statement, 2, (model.lastOrderTotal ?? 0).getRoundFigure())
                sqlite3_bind_int64(statement, 3, sqlite3_int64(0))
                sqlite3_bind_int64(statement, 4, sqlite3_int64(model.tableStatusId ?? 0))
                sqlite3_bind_int64(statement, 5, sqlite3_int64(model.lastOrderId ?? 0))
                sqlite3_bind_int64(statement, 6, sqlite3_int64(model.locked ? 1: 0))
                sqlite3_bind_int64(statement, 7, sqlite3_int64(model.mergeTableId))
                sqlite3_bind_int64(statement, 8, sqlite3_int64(model.updaterId))
                
                if sqlite3_step(statement) == SQLITE_DONE {
                } else {
                    let errmsg = String(cString: sqlite3_errmsg(database))
                    print("Could not update table, \(errmsg)")
                }
                sqlite3_finalize(statement)
            }
        }
        sqlite3_close(database)
    }
    
    //MARK: Orders
    static func FetchOrdersForTable(table:TableModel)->[OrderModel]{
        return FetchOrders("SELECT * FROM Orders WHERE isDelete = 0 AND order_status_id IN (1,2,12,13,14,4,7,6) AND business_id = \(AppConstants.businessData?.id ?? 0) AND table_id = \(table.id ?? 0)")
    }
    
    static func FetchOrders(_ str:String, _ db:OpaquePointer? = nil) -> [OrderModel]{
        var result = [OrderModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if db != nil{
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = OrderModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.customerId = Int(sqlite3_column_int64(stmt, 1))
                    mod.comment = String(cString: sqlite3_column_text(stmt, 2))
                    mod.customerName = String(cString: sqlite3_column_text(stmt, 3))
                    mod.deliveryCharge = sqlite3_column_double(stmt, 4)
                    let timstamp = sqlite3_column_int64(stmt, 5)
                    mod.deliveryDate = (Date.init(timeIntervalSince1970: Double(timstamp))).getDateInString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
                    mod.discount = sqlite3_column_double(stmt, 6)
                    mod.distance = sqlite3_column_double(stmt, 7)
                    mod.gratuity = sqlite3_column_double(stmt, 8)
                    mod.isUpdated = Int(sqlite3_column_int64(stmt, 9))
                    mod.isDeleted = Int(sqlite3_column_int64(stmt, 10))
                    mod.isNewOrder = Int(sqlite3_column_int64(stmt, 11))
                    mod.noGuest = Int(sqlite3_column_int64(stmt, 12))
                    mod.orderActionId = Int(sqlite3_column_int64(stmt, 13))
                    mod.orderStatusId = Int(sqlite3_column_int64(stmt, 15))
                    if mod.orderStatusId ?? 0 != 0{
                        mod.orderStatus = fetchOrderStatuseById(id: mod.orderStatusId!)
                    }
                    mod.orderType = String(cString: sqlite3_column_text(stmt, 16))
                    mod.orderTypeId = Int(sqlite3_column_int64(stmt, 17))
                    mod.serviceCharge = sqlite3_column_double(stmt, 18)
                    mod.splitCount = Int(sqlite3_column_int64(stmt, 19))
                    mod.subTotal = sqlite3_column_double(stmt, 20)
                    mod.tableId = Int(sqlite3_column_int64(stmt, 21))
                    mod.tax = sqlite3_column_double(stmt, 22)
                    mod.total = sqlite3_column_double(stmt, 23).getRoundFigure()
                    mod.totalPaid = sqlite3_column_double(stmt, 24)
                    mod.updaterId = Int(sqlite3_column_int64(stmt, 25))
                    mod.calculateLoyaltyAmount = sqlite3_column_double(stmt, 27)
                    mod.calculateLoyaltyPoint = sqlite3_column_double(stmt, 28)
                    mod.isArchived = Int(sqlite3_column_int64(stmt, 29)) == 1
                    mod.uniqueId = String(cString: sqlite3_column_text(stmt, 32))
                    let timestamp = Double(sqlite3_column_int64(stmt, 34))
                    if timestamp > 0{
                        mod.createdDate = Date.init(timeIntervalSinceReferenceDate: timestamp)
                    }
                    mod.customer = CoreDataHelper.shared.fetchCustomerObect(id: mod.customerId ?? 0)
                    if mod.tableId ?? 0 != 0{
                        mod.table = fetchTableDataFromId(id: mod.tableId ?? 0)
                    }
                    mod.arrProducts = FetchOrderProducts("SELECT * FROM OrderProducts WHERE order_id = \(mod.id ?? 0) ORDER BY updated_at DESC", db)
                    
                    mod.arrOrderSplit = FetchOrderGroups("SELECT * FROM OrderGroup WHERE order_id = \(mod.id ?? 0)", db, orderId: mod.id ?? 0)
                    mod.arrOrderPayments = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_id = \(mod.id ?? 0)", db)
                    mod._id = Int(sqlite3_column_int64(stmt, 35))
                    mod.isAutoServiceChargeRemove = Int(sqlite3_column_int64(stmt, 36)) == 1
                    if let cString = sqlite3_column_text(stmt, 37) {
                        mod.displayOrderId = String(cString: cString)
                    }
                    mod.deliveryCollectionTimeSlot = String(cString: sqlite3_column_text(stmt, 38))
                    mod.isDeliveryCollectionAsk = Int(sqlite3_column_int64(stmt, 39)) == 1
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }else{
            
            let DBpath:String = copyDatabaseToDirectory()
            
            var db: OpaquePointer? = nil
            if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
                if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                    while (sqlite3_step(stmt) == SQLITE_ROW){
                        let mod = OrderModel()
                        
                        mod.id = Int(sqlite3_column_int64(stmt, 0))
                        mod.customerId = Int(sqlite3_column_int64(stmt, 1))
                        mod.comment = String(cString: sqlite3_column_text(stmt, 2))
                        mod.customerName = String(cString: sqlite3_column_text(stmt, 3))
                        mod.deliveryCharge = sqlite3_column_double(stmt, 4)
                        let timstamp = sqlite3_column_int64(stmt, 5)
                        mod.deliveryDate = (Date.init(timeIntervalSince1970: Double(timstamp))).getDateInString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'")
                        mod.discount = sqlite3_column_double(stmt, 6)
                        mod.distance = sqlite3_column_double(stmt, 7)
                        mod.gratuity = sqlite3_column_double(stmt, 8)
                        mod.isUpdated = Int(sqlite3_column_int64(stmt, 9))
                        mod.isDeleted = Int(sqlite3_column_int64(stmt, 10))
                        mod.isNewOrder = Int(sqlite3_column_int64(stmt, 11))
                        mod.noGuest = Int(sqlite3_column_int64(stmt, 12))
                        mod.orderActionId = Int(sqlite3_column_int64(stmt, 13))
                        mod.orderStatusId = Int(sqlite3_column_int64(stmt, 15))
                        if mod.orderStatusId ?? 0 != 0{
                            mod.orderStatus = fetchOrderStatuseById(id: mod.orderStatusId!)
                        }
                        
                        mod.orderType = String(cString: sqlite3_column_text(stmt, 16))
                        mod.orderTypeId = Int(sqlite3_column_int64(stmt, 17))
                        mod.serviceCharge = sqlite3_column_double(stmt, 18)
                        mod.splitCount = Int(sqlite3_column_int64(stmt, 19))
                        mod.subTotal = sqlite3_column_double(stmt, 20)
                        mod.tableId = Int(sqlite3_column_int64(stmt, 21))
                        mod.tax = sqlite3_column_double(stmt, 22)
                        mod.total = sqlite3_column_double(stmt, 23).getRoundFigure()
                        mod.totalPaid = sqlite3_column_double(stmt, 24)
                        mod.updaterId = Int(sqlite3_column_int64(stmt, 25))
                        mod.calculateLoyaltyAmount = sqlite3_column_double(stmt, 27)
                        mod.calculateLoyaltyPoint = sqlite3_column_double(stmt, 28)
                        mod.isArchived = Int(sqlite3_column_int64(stmt, 29)) == 1
                        mod.uniqueId = String(cString: sqlite3_column_text(stmt, 32))
                        //Int(sqlite3_column_int64(stmt, 32))
                        let timestamp = Double(sqlite3_column_int64(stmt, 34))
                        if timestamp > 0{
                            mod.createdDate = Date.init(timeIntervalSinceReferenceDate: timestamp)
                        }
                        //print("mod.uniqueId2",mod.uniqueId)
                        mod.customer = CoreDataHelper.shared.fetchCustomerObect(id: mod.customerId ?? 0)
                        if mod.tableId ?? 0 != 0{
                            mod.table = fetchTableDataFromId(id: mod.tableId ?? 0)
                        }
                        mod.arrProducts = FetchOrderProducts("SELECT * FROM OrderProducts WHERE order_id = \(mod.id ?? 0) ORDER BY updated_at DESC", db)
                        
                        mod.arrOrderSplit = FetchOrderGroups("SELECT * FROM OrderGroup WHERE order_id = \(mod.id ?? 0)", db, orderId: mod.id ?? 0)
                        mod.arrOrderPayments = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_id = \(mod.id ?? 0)", db)
                        mod._id = Int(sqlite3_column_int64(stmt, 35))
                        mod.isAutoServiceChargeRemove = Int(sqlite3_column_int64(stmt, 36)) == 1
                        if let cString = sqlite3_column_text(stmt, 37) {
                            mod.displayOrderId = String(cString: cString)
                        }
                        mod.deliveryCollectionTimeSlot = String(cString: sqlite3_column_text(stmt, 38))
                        mod.isDeliveryCollectionAsk = Int(sqlite3_column_int64(stmt, 39)) == 1
                        result.append(mod)
                    }
                    sqlite3_finalize(stmt)
                }
            }
            
            sqlite3_close(db)
        }
        return result
    }
    
    static func fetchOngoingOrdersData(orderType:Int, isArchive:Int = 0)->[OrderModel]{
        return FetchOrders("SELECT * FROM Orders WHERE order_type_id = \(orderType) AND isDelete = 0 AND order_status_id IN (1,2,12,13,14,4,7,6) AND business_id = \(AppConstants.businessData?.id ?? 0) AND is_archived = \(isArchive) ORDER BY id DESC")
    }
    
    static func fetchCompletedOrdersData(orderType:Int, isArchive:Int = 0)->[OrderModel]{
        let date = Int(((Date().startOfDate() ?? Date()).timeIntervalSince1970))
        return FetchOrders("SELECT * FROM Orders WHERE order_type_id = \(orderType) AND isDelete = 0 AND delivery_date >= \(date) AND order_status_id IN (5,10,9,20,3,19) AND business_id = \(AppConstants.businessData?.id ?? 0) AND is_archived = \(isArchive) ORDER BY id DESC")
    }
    
    static func fetchAllOrdersData(orderType:Int, isArchive:Int = 0)->[OrderModel]{
        let date = Int(((Date().startOfDate() ?? Date()).timeIntervalSince1970))
        return FetchOrders("SELECT * FROM Orders WHERE order_type_id = \(orderType) AND isDelete = 0 AND ((order_status_id IN (1,2,12,13,14,4,7,6)) OR (delivery_date >= \(date) AND order_status_id  IN (5,10,9,20,3,19))) AND business_id = \(AppConstants.businessData?.id ?? 0) AND is_archived = \(isArchive) ORDER BY id DESC")
    }
    
    static func fetchOrdersDataFromDates(startDate:Date? = nil,endDate:Date? = nil, isArchive:Int = 0, table:TableModel? = nil)->[OrderModel]{
        var strTable = ""
        if table != nil{
            strTable = "AND table_id = \(table!.id ?? 0)"
        }
        if startDate != nil && endDate != nil{
            return FetchOrders("SELECT * FROM Orders WHERE isDelete = 0 AND delivery_date >= \(Int64(startDate!.timeIntervalSince1970)) AND delivery_date < \(Int64(endDate!.timeIntervalSince1970)) AND business_id = \(AppConstants.businessData?.id ?? 0) AND is_archived = \(isArchive) \(strTable) ORDER BY id DESC")
        }else{
            return FetchOrders("SELECT * FROM Orders WHERE isDelete = 0 AND business_id = \(AppConstants.businessData?.id ?? 0) AND is_archived = \(isArchive) \(strTable) ORDER BY id DESC")
        }
    }
    
    static func fetchOfflineOrdersData()->[OrderModel]{
        return FetchOrders("SELECT * FROM Orders WHERE is_update = 1 AND business_id = \(AppConstants.businessData?.id ?? 0) ORDER BY delivery_date")
    }
    
    static func fetchFailedOrdersData()->[OrderModel]{
        return FetchOrders("SELECT * FROM Orders WHERE failed = 1 AND business_id = \(AppConstants.businessData?.id ?? 0)")
    }
    
    static func fetchOrderDataById(orderId:Int)->OrderModel?{
        return FetchOrders("SELECT * FROM Orders WHERE id = \(orderId) AND business_id = \(AppConstants.businessData?.id ?? 0) LIMIT 1").first
    }
    
    static func fetchOrderDataByUniqueId(orderId:String)->OrderModel?{
        return FetchOrders("SELECT * FROM Orders WHERE unique_id = \(orderId) AND business_id = \(AppConstants.businessData?.id ?? 0) LIMIT 1").first
    }
    
    static func fetchLastOrder()->OrderModel?{
        return FetchOrders("SELECT * FROM Orders ORDER BY _id DESC LIMIT 1").first
    }
    
    static func fetchOrderForNotification(orderId:Int)->OrderModel?{
        if let order = FetchOrders("SELECT * FROM Orders WHERE id = \(orderId) AND business_id = \(AppConstants.businessData?.id ?? 0) LIMIT 1").first{
            return order
        }else if let order = FetchOrders("SELECT * FROM Orders WHERE offline_id = \(orderId) AND business_id = \(AppConstants.businessData?.id ?? 0) LIMIT 1").first{
            return order
        }
        return nil
    }
    
    static func deleteOrderItem(productId:Int, orderId:Int){
        if orderId != 0{
            _ = ExecuteQuery("UPDATE OrderProducts SET isDelete = 1 WHERE id = \(productId)")
        }else{
            _ = ExecuteQuery("DELETE FROM OrderProducts WHERE id = \(productId)")
            _ = ExecuteQuery("DELETE FROM OrderIngredients WHERE product_id = \(productId)")
            _ = ExecuteQuery("DELETE FROM OrderAddons WHERE product_id = \(productId)")
        }
    }
    
    static func updateOrderFailedStatus(failed:Int, orderId:String){
        _ = ExecuteQuery("UPDATE Orders SET failed = \(failed) WHERE unique_id = \(orderId)")
        NotificationCenter.default.post(name: AppConstants.notifications.kFailedOrder, object: nil)
    }
    
    static func SaveOrderIntoDatabase(arrOrder:[OrderModel], shouldDelete:Bool = true, showMessage:Bool = false, isFromSplash:Bool = false, offlineId:Int? = nil){
        let dbPath = copyDatabaseToDirectory()
        
        var strQuery = "SELECT * FROM Orders WHERE business_id = \(AppConstants.businessData?.id ?? 0)"
        if !isFromSplash{
            let date = Date().addingTimeInterval(-24*60*60)
            let timestamp = Int((date.startOfDate() ?? date).timeIntervalSince1970)
            strQuery = "SELECT * FROM Orders WHERE delivery_date >= \(timestamp) AND business_id = \(AppConstants.businessData?.id ?? 0)"
        }
        
        var database: OpaquePointer? = nil
        let insert = "INSERT INTO Orders (id, customer_id, comment, customer_name, delivery_charge, delivery_date, discount, distance, gratuity, is_update, isDelete, isNewOrder, no_guest, order_action_id, order_status, order_status_id, order_type, order_type_id, service_charge, split_count, sub_total, table_id, tax, total, total_paid, updater_id, business_id, calculate_loyalty_amount, calculate_loyalty_point, is_archived, unique_id, created_date, _id, is_auto_service_charge_remove, display_order_id, delivery_collection_time_slot, is_delivery_collection_ask) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        
        let update = "UPDATE Orders SET customer_id = ?, comment = ?, customer_name = ?, delivery_charge = ?, delivery_date = ?, discount = ?, distance = ?, gratuity = ?, is_update = ?, isDelete = ?, isNewOrder = ?, no_guest = ?, order_action_id = ?, order_status = ?, order_status_id = ?, order_type = ?, order_type_id = ?, service_charge = ?, split_count = ?, sub_total = ?, table_id = ?, tax = ?, total = ?, total_paid = ?, updater_id = ?, business_id = ?, calculate_loyalty_amount = ?, calculate_loyalty_point = ?, is_archived = ?, id = ?, created_date = ?, is_auto_service_charge_remove = ?, display_order_id = ?, delivery_collection_time_slot = ?, is_delivery_collection_ask = ?  WHERE unique_id = ?"
        
        
        if (sqlite3_open(dbPath, &database) == SQLITE_OK) {
            
            let arrOfflineOrder = FetchOrders(strQuery, database)
            var arrUpdateOrder = [OrderModel]()
            var _id = (arrOfflineOrder.last?._id ?? 0) + 1
            for order in arrOfflineOrder {
                if let _ = arrOrder.filter({ (model) -> Bool in
                    return model.uniqueId == order.uniqueId
                }).first{
                    arrUpdateOrder.append(order)
                }else{
                    if shouldDelete{
                        if order.isNewOrder == 0{
                            _ = ExecuteQuery("DELETE FROM Orders WHERE id = \(order.id ?? 0) AND isNewOrder = 0",database)
                            _ = ExecuteQuery("DELETE FROM OrderProducts WHERE order_id = \(order.id ?? 0)",database)
                            _ = ExecuteQuery("DELETE FROM OrderAddons WHERE order_id = \(order.id ?? 0)",database)
                            _ = ExecuteQuery("DELETE FROM OrderIngredients WHERE order_id = \(order.id ?? 0)",database)
                            _ = ExecuteQuery("DELETE FROM OrderGroup WHERE order_id = \(order.id ?? 0)",database)
                            _ = ExecuteQuery("DELETE FROM OrderPayment WHERE order_id = \(order.id ?? 0)",database)
                        }
                    }
                }
            }
            
            for i in 0..<arrOrder.count {
                let order = arrOrder[i]
                if showMessage{
                    AppCommonMethods.changeProgressMessage(msg: "Saving orders\n\(i+1)/\(arrOrder.count)")
                }
                let index = arrUpdateOrder.firstIndex(where: { (model) -> Bool in
                    return model.id ?? 0 == order.id ?? 0
                })
                if index == nil{
                    let uniqueOrder = fetchOrderDataByUniqueId(orderId: order.uniqueId)
                    if uniqueOrder == nil{
                        self.insertOrderProducts(orderId: order.id ?? 0, arrProducts: order.arrProducts, database: database, isFromList: true, checkUpdated: false)
                        if order.arrOrderPayments.count > 0{
                            self.insertOrderPayments(orderId: order.id ?? 0, arrPayments: order.arrOrderPayments, database: database, isFromList: true)
                        }
                        if order.arrOrderSplit.count > 0{
                            self.insertOrderGroups(orderId: order.id ?? 0, arrGroups: order.arrOrderSplit, database: database)
                        }
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(order.id ?? 0))
//                            print(order.customerId ?? 0)
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(order.customerId ?? 0))
                            sqlite3_bind_text(statement, 3, ((order.comment ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 4, ((order.customerName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 5, order.deliveryCharge ?? 0)
                            let date = (order.deliveryDate ?? "").getDateFromString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'") ?? Date()
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(date.timeIntervalSince1970))
                            sqlite3_bind_double(statement, 7, order.discount ?? 0)
                            sqlite3_bind_double(statement, 8, order.distance ?? 0)
                            sqlite3_bind_double(statement, 9, order.gratuity ?? 0)
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(order.noGuest ?? 0))
                            sqlite3_bind_int64(statement, 14, sqlite3_int64(order.orderActionId ?? 0))
                            sqlite3_bind_text(statement, 15, ((order.orderStatus?.status ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 16, sqlite3_int64(order.orderStatusId ?? 0))
                            sqlite3_bind_text(statement, 17, ((order.orderType ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 18, sqlite3_int64(order.orderTypeId ?? 0))
                            sqlite3_bind_double(statement, 19, order.serviceCharge ?? 0)
                            sqlite3_bind_int64(statement, 20, sqlite3_int64(order.splitCount ?? 0))
                            sqlite3_bind_double(statement, 21, (order.subTotal ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 22, sqlite3_int64(order.tableId ?? 0))
                            sqlite3_bind_double(statement, 23, order.tax ?? 0)
                            sqlite3_bind_double(statement, 24, (order.total ?? 0).getRoundFigure())
                            sqlite3_bind_double(statement, 25, (order.totalPaid ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 26, sqlite3_int64(order.updaterId ?? 0))
                            
                            sqlite3_bind_int64(statement, 27, sqlite3_int64(AppConstants.businessData?.id ?? 0))
                            sqlite3_bind_double(statement, 28, order.calculateLoyaltyAmount ?? 0)
                            sqlite3_bind_double(statement, 29, order.calculateLoyaltyPoint ?? 0)
                            sqlite3_bind_int64(statement, 30, sqlite3_int64(order.isArchived ? 1 : 0))
                            sqlite3_bind_text(statement, 31, ((order.uniqueId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            //sqlite3_bind_int64(statement, 31, sqlite3_int64(order.uniqueId))
                            sqlite3_bind_int64(statement, 32, sqlite3_int64(order.createdDate?.timeIntervalSinceReferenceDate ?? 0))
                            _id += 1
                            sqlite3_bind_int64(statement, 33, sqlite3_int64(_id))
                            sqlite3_bind_int64(statement, 34, sqlite3_int64(order.isAutoServiceChargeRemove ? 1 : 0))
                            sqlite3_bind_text(statement, 35, ((order.displayOrderId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 36, ((order.deliveryCollectionTimeSlot) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 37, sqlite3_int64(order.isDeliveryCollectionAsk ? 1 : 0))
//                            print("order.uniqueId1",order.uniqueId)
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not insert order, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }else{
                    if arrUpdateOrder[index!].isUpdated == 0 || AppConstants.DataSyncMode == "Auto"{
                        self.insertOrderProducts(orderId: order.id ?? 0, arrProducts: order.arrProducts, database: database, isFromList: true, checkUpdated: false)
                        if order.arrOrderPayments.count > 0{
                            self.insertOrderPayments(orderId: order.id ?? 0, arrPayments: order.arrOrderPayments, database: database, isFromList: true)
                        }
                        if order.arrOrderSplit.count > 0{
                            self.insertOrderGroups(orderId: order.id ?? 0, arrGroups: order.arrOrderSplit, database: database)
                        }
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
//                            print(order.customerId ?? 0)
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(order.customerId ?? 0))
                            sqlite3_bind_text(statement, 2, ((order.comment ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 3, ((order.customerName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 4, order.deliveryCharge ?? 0)
                            let date = (order.deliveryDate ?? "").getDateFromString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'") ?? Date()
                            sqlite3_bind_int64(statement, 5, sqlite3_int64(date.timeIntervalSince1970))
                            sqlite3_bind_double(statement, 6, order.discount ?? 0)
                            sqlite3_bind_double(statement, 7, order.distance ?? 0)
                            sqlite3_bind_double(statement, 8, order.gratuity ?? 0)
                            sqlite3_bind_int64(statement, 9, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(0))
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(order.noGuest ?? 0))
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(order.orderActionId ?? 0))
                            sqlite3_bind_text(statement, 14, ((order.orderStatus?.status ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 15, sqlite3_int64(order.orderStatusId ?? 0))
                            sqlite3_bind_text(statement, 16, ((order.orderType ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 17, sqlite3_int64(order.orderTypeId ?? 0))
                            sqlite3_bind_double(statement, 18, order.serviceCharge ?? 0)
                            sqlite3_bind_int64(statement, 19, sqlite3_int64(order.splitCount ?? 0))
                            sqlite3_bind_double(statement, 20, (order.subTotal ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 21, sqlite3_int64(order.tableId ?? 0))
                            sqlite3_bind_double(statement, 22, order.tax ?? 0)
                            sqlite3_bind_double(statement, 23, (order.total ?? 0).getRoundFigure())
                            sqlite3_bind_double(statement, 24, (order.totalPaid ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 25, sqlite3_int64(order.updaterId ?? 0))
                            sqlite3_bind_int64(statement, 26, sqlite3_int64(AppConstants.businessData?.id ?? 0))
                            sqlite3_bind_double(statement, 27, order.calculateLoyaltyAmount ?? 0)
                            sqlite3_bind_double(statement, 28, order.calculateLoyaltyPoint ?? 0)
                            sqlite3_bind_int64(statement, 29, sqlite3_int64(order.isArchived ? 1 : 0))
                            sqlite3_bind_int64(statement, 30, sqlite3_int64(order.id ?? 0))
                            sqlite3_bind_int64(statement, 31, sqlite3_int64(order.createdDate?.timeIntervalSinceReferenceDate ?? 0))
                            sqlite3_bind_int64(statement, 32, sqlite3_int64(order.isAutoServiceChargeRemove ? 1 : 0))
                            sqlite3_bind_text(statement, 33, ((order.displayOrderId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 34, ((order.deliveryCollectionTimeSlot) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 35, sqlite3_int64(order.isDeliveryCollectionAsk ? 1 : 0))
                            sqlite3_bind_text(statement, 36, ((order.uniqueId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            //sqlite3_bind_int64(statement, 32, sqlite3_int64(order.uniqueId))
//                            print("order.uniqueId2",order.uniqueId)
                            if sqlite3_step(statement) == SQLITE_DONE {
                                
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not update order, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }
                
                if order.customer != nil{
                    CoreDataHelper.shared.saveAndUpdateCusomer(customers: [order.customer!], offline: 0)
                }
            }
        }
        sqlite3_close(database)
    }
    
    static func UpdateOrderFailed(id:Int){
        _ = ExecuteQuery("UPDATE Orders SET is_update = 1, failed = 1 WHERE unique_id = \(id)")
    }
    
    static func SaveOrderIntoDatabase(order:OrderModel, updated:Int, isDeleted:Int? = nil, isFromAPI:Bool = false){
        let dbPath = copyDatabaseToDirectory()
//        print("Order updated", updated)
        var database: OpaquePointer? = nil
        let insert = "INSERT INTO Orders (id, customer_id, comment, customer_name, delivery_charge, delivery_date, discount, distance, gratuity, is_update, isDelete, isNewOrder, no_guest, order_action_id, order_status, order_status_id, order_type, order_type_id, service_charge, split_count, sub_total, table_id, tax, total, total_paid, updater_id, business_id, calculate_loyalty_amount, calculate_loyalty_point, is_archived, unique_id, failed, created_date, _id, is_auto_service_charge_remove, display_order_id, delivery_collection_time_slot, is_delivery_collection_ask) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        
        let update = "UPDATE Orders SET customer_id = ?, comment = ?, customer_name = ?, delivery_charge = ?, delivery_date = ?, discount = ?, distance = ?, gratuity = ?, is_update = ?, isDelete = ?, isNewOrder = ?, no_guest = ?, order_action_id = ?, order_status = ?, order_status_id = ?, order_type = ?, order_type_id = ?, service_charge = ?, split_count = ?, sub_total = ?, table_id = ?, tax = ?, total = ?, total_paid = ?, updater_id = ?, business_id = ?, calculate_loyalty_amount = ?, calculate_loyalty_point = ?, is_archived = ?, id = ?, failed = ?, created_date = ?, is_auto_service_charge_remove = ?, display_order_id = ?, delivery_collection_time_slot = ?, is_delivery_collection_ask = ?  WHERE unique_id = ?"
        
        
        if (sqlite3_open(dbPath, &database) == SQLITE_OK) {
            self.insertOrderProducts(orderId: order.id ?? 0, arrProducts: order.arrProducts, database: database, checkUpdated: !isFromAPI)
            if order.arrOrderPayments.count > 0{
                self.insertOrderPayments(orderId: order.id ?? 0, arrPayments: order.arrOrderPayments, database: database, isFromList: true)
            }
            if order.arrOrderSplit.count > 0{
                self.insertOrderGroups(orderId: order.id ?? 0, arrGroups: order.arrOrderSplit, database: database, updateOffline: isFromAPI)
            }
            if fetchOrderDataByUniqueId(orderId: order.uniqueId) != nil{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
//                    print(order.customerId ?? 0)
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(order.customerId ?? 0))
                    sqlite3_bind_text(statement, 2, ((order.comment ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 3, ((order.customerName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_double(statement, 4, order.deliveryCharge ?? 0)
                    let date = (order.deliveryDate ?? "").getDateFromString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'") ?? Date()
                    sqlite3_bind_int64(statement, 5, sqlite3_int64(date.timeIntervalSince1970))
                    sqlite3_bind_double(statement, 6, order.discount ?? 0)
                    sqlite3_bind_double(statement, 7, order.distance ?? 0)
                    sqlite3_bind_double(statement, 8, order.gratuity ?? 0)
                    sqlite3_bind_int64(statement, 9, sqlite3_int64(updated))
                    if isDeleted != nil{
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(isDeleted!))
                    }else{
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(order.isDeleted ))
                    }
                    sqlite3_bind_int64(statement, 11, sqlite3_int64(order.isNewOrder))
                    sqlite3_bind_int64(statement, 12, sqlite3_int64(order.noGuest ?? 0))
                    sqlite3_bind_int64(statement, 13, sqlite3_int64(order.orderActionId ?? 0))
                    sqlite3_bind_text(statement, 14, ((order.orderStatus?.status ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 15, sqlite3_int64(order.orderStatusId ?? 0))
                    sqlite3_bind_text(statement, 16, ((order.orderType ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 17, sqlite3_int64(order.orderTypeId ?? 0))
                    sqlite3_bind_double(statement, 18, order.serviceCharge ?? 0)
                    sqlite3_bind_int64(statement, 19, sqlite3_int64(order.splitCount ?? 0))
                    sqlite3_bind_double(statement, 20, (order.subTotal ?? 0).getRoundFigure())
                    sqlite3_bind_int64(statement, 21, sqlite3_int64(order.tableId ?? 0))
                    sqlite3_bind_double(statement, 22, order.tax ?? 0)
                    sqlite3_bind_double(statement, 23, (order.total ?? 0).getRoundFigure())
                    sqlite3_bind_double(statement, 24, (order.totalPaid ?? 0).getRoundFigure())
                    if AppConstants.userData != nil{
                        sqlite3_bind_int64(statement, 25, sqlite3_int64(AppConstants.userData?.id ?? 0))
                    }else{
                        sqlite3_bind_int64(statement, 25, sqlite3_int64(order.updaterId ?? 0))
                    }
                    sqlite3_bind_int64(statement, 26, sqlite3_int64(AppConstants.businessData?.id ?? 0))
                    sqlite3_bind_double(statement, 27, order.calculateLoyaltyAmount ?? 0)
                    sqlite3_bind_double(statement, 28, order.calculateLoyaltyPoint ?? 0)
                    sqlite3_bind_int64(statement, 29, sqlite3_int64(order.isArchived ? 1 : 0))
                    sqlite3_bind_int64(statement, 30, sqlite3_int64(order.id ?? 0))
                    sqlite3_bind_int64(statement, 31, sqlite3_int64(0))
                    sqlite3_bind_int64(statement, 32, sqlite3_int64(order.createdDate?.timeIntervalSinceReferenceDate ?? 0))
                    sqlite3_bind_int64(statement, 33, sqlite3_int64(order.isAutoServiceChargeRemove ? 1 : 0))
                    sqlite3_bind_text(statement, 34, ((order.displayOrderId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 35, ((order.deliveryCollectionTimeSlot) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 36, sqlite3_int64(order.isDeliveryCollectionAsk ? 1 : 0))
                    sqlite3_bind_text(statement, 37, ((order.uniqueId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    //sqlite3_bind_int64(statement, 33, sqlite3_int64(order.uniqueId))
//                    print("order.uniqueId3",order.uniqueId)
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not updated order, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }else{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(order.id ?? 0))
//                    print(order.customerId ?? 0)
                    sqlite3_bind_int64(statement, 2, sqlite3_int64(order.customerId ?? 0))
                    sqlite3_bind_text(statement, 3, ((order.comment ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 4, ((order.customerName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_double(statement, 5, order.deliveryCharge ?? 0)
                    let date = (order.deliveryDate ?? "").getDateFromString(format: "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'") ?? Date()
                    sqlite3_bind_int64(statement, 6, sqlite3_int64(date.timeIntervalSince1970))
                    sqlite3_bind_double(statement, 7, order.discount ?? 0)
                    sqlite3_bind_double(statement, 8, order.distance ?? 0)
                    sqlite3_bind_double(statement, 9, order.gratuity ?? 0)
                    sqlite3_bind_int64(statement, 10, sqlite3_int64(updated))
                    if isDeleted != nil{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(isDeleted!))
                    }else{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(order.isDeleted ))
                    }
                    sqlite3_bind_int64(statement, 12, sqlite3_int64(order.isNewOrder))
                    sqlite3_bind_int64(statement, 13, sqlite3_int64(order.noGuest ?? 0))
                    sqlite3_bind_int64(statement, 14, sqlite3_int64(order.orderActionId ?? 0))
                    sqlite3_bind_text(statement, 15, ((order.orderStatus?.status ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 16, sqlite3_int64(order.orderStatusId ?? 0))
                    sqlite3_bind_text(statement, 17, ((order.orderType ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 18, sqlite3_int64(order.orderTypeId ?? 0))
                    sqlite3_bind_double(statement, 19, order.serviceCharge ?? 0)
                    sqlite3_bind_int64(statement, 20, sqlite3_int64(order.splitCount ?? 0))
                    sqlite3_bind_double(statement, 21, (order.subTotal ?? 0).getRoundFigure())
                    sqlite3_bind_int64(statement, 22, sqlite3_int64(order.tableId ?? 0))
                    sqlite3_bind_double(statement, 23, order.tax ?? 0)
                    sqlite3_bind_double(statement, 24, (order.total ?? 0).getRoundFigure())
                    sqlite3_bind_double(statement, 25, (order.totalPaid ?? 0).getRoundFigure())
                    if AppConstants.userData != nil{
                        sqlite3_bind_int64(statement, 26, sqlite3_int64(AppConstants.userData?.id ?? 0))
                    }else{
                        sqlite3_bind_int64(statement, 26, sqlite3_int64(order.updaterId ?? 0))
                    }
                    sqlite3_bind_int64(statement, 27, sqlite3_int64(AppConstants.businessData?.id ?? 0))
                    sqlite3_bind_double(statement, 28, order.calculateLoyaltyAmount ?? 0)
                    sqlite3_bind_double(statement, 29, order.calculateLoyaltyPoint ?? 0)
                    sqlite3_bind_int64(statement, 30, sqlite3_int64(order.isArchived ? 1 : 0))
                    sqlite3_bind_text(statement, 31, ((order.uniqueId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    //sqlite3_bind_int64(statement, 31, sqlite3_int64(order.uniqueId))
                    sqlite3_bind_int64(statement, 32, sqlite3_int64(0))
                    sqlite3_bind_int64(statement, 33, sqlite3_int64(order.createdDate?.timeIntervalSinceReferenceDate ?? 0))
                    let id = (fetchLastOrder()?._id ?? 0)+1
                    sqlite3_bind_int64(statement, 34, sqlite3_int64(id))
                    sqlite3_bind_int64(statement, 35, sqlite3_int64(order.isAutoServiceChargeRemove ? 1 : 0))
                    sqlite3_bind_text(statement, 36, ((order.displayOrderId) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 37, ((order.deliveryCollectionTimeSlot) as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 38, sqlite3_int64(order.isDeliveryCollectionAsk ? 1 : 0))
//                    print("order.uniqueId4",order.uniqueId)
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert order, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }
            
        }
        sqlite3_close(database)
    }
    
    static func updateOrderStatus(id:Int, statusId:Int){
        _ = ExecuteQuery("UPDATE Orders SET order_status_id = \(statusId) WHERE id = \(id)")
    }
    
    static func deleteOrder(orderId:Int,isFromAPI:Bool){
        if isFromAPI{
            _ = ExecuteQuery("DELETE FROM Orders WHERE id = \(orderId)")
            _ = ExecuteQuery("DELETE FROM OrderProducts WHERE order_id = \(orderId)")
            _ = ExecuteQuery("DELETE FROM OrderIngredients WHERE order_id = \(orderId)")
            _ = ExecuteQuery("DELETE FROM OrderAddons WHERE order_id = \(orderId)")
            _ = ExecuteQuery("DELETE FROM OrderGroup WHERE order_id = \(orderId)")
            _ = ExecuteQuery("DELETE FROM OrderPayment WHERE order_id = \(orderId)")
        }else{
            _ = ExecuteQuery("UPDATE Orders SET isDelete = 1 WHERE id = \(orderId)")
        }
    }
    
    //MARK: Order Products
    static func FetchOrderProducts(_ str:String,_ db:OpaquePointer?) -> [OrderProductModel]{
        var result = [OrderProductModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        let DBpath:String = copyDatabaseToDirectory()
        
        
        if db != nil{
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = OrderProductModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.orderId = Int(sqlite3_column_int64(stmt, 1))
                    mod.orderSplitId = Int(sqlite3_column_int64(stmt, 2))
                    mod.prepLocationId = Int(sqlite3_column_int64(stmt, 3))
                    mod.price = sqlite3_column_double(stmt, 4)
                    mod.prodcutId = Int(sqlite3_column_int64(stmt, 5))
                    mod.productName = String(cString: sqlite3_column_text(stmt, 6))
                    mod.quantity = Int(sqlite3_column_int64(stmt, 7))
                    mod.specialInstruction = String(cString: sqlite3_column_text(stmt, 8))
                    mod.subTotal = sqlite3_column_double(stmt, 9)
                    mod.updaterId = Int(sqlite3_column_int64(stmt, 10))
                    mod.groupOffline = Int(sqlite3_column_int64(stmt, 11))
                    mod.instructionPrice = sqlite3_column_double(stmt, 12)
                    mod.isItemUpdated = Int(sqlite3_column_int64(stmt, 14))
                    mod.misc = Int(sqlite3_column_int64(stmt, 15))
                    mod.sentToKitchen = Int(sqlite3_column_int64(stmt, 16)) == 1
                    mod.groupUniqueId = String(cString: sqlite3_column_text(stmt, 19))
                    //Int(sqlite3_column_int64(stmt, 19))
                    mod.uniqueId = String(cString: sqlite3_column_text(stmt, 18))
                    //Int64(sqlite3_column_int64(stmt, 18))
                    mod.isDelete = Int(sqlite3_column_int64(stmt, 13)) == 1
                    mod.arrAddons = FetchOrderAddons("SELECT * FROM OrderAddons WHERE product_id = \(mod.id ?? 0)", db)
                    mod.arrIngriedent = FetchOrderIngridents("SELECT * FROM OrderIngredients WHERE product_id = \(mod.id ?? 0)", db)
                    mod.product = CoreDataHelper.shared.fetchProductObejct(prodcutId: mod.prodcutId ?? 0)
                    mod.updatedAt = String(cString: sqlite3_column_text(stmt, 20))
                    mod.sentToKitchenQuantity = Int(sqlite3_column_int64(stmt, 21))
                    mod.total = sqlite3_column_double(stmt, 22)
                    mod.addonPrice = sqlite3_column_double(stmt, 23)
                    mod.ingredientsPrice = sqlite3_column_double(stmt, 24)
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }else{
            var db: OpaquePointer? = nil
            if (sqlite3_open(DBpath, &db)==SQLITE_OK){
                if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                    while (sqlite3_step(stmt) == SQLITE_ROW){
                        let mod = OrderProductModel()
                        
                        mod.id = Int(sqlite3_column_int64(stmt, 0))
                        mod.orderId = Int(sqlite3_column_int64(stmt, 1))
                        mod.orderSplitId = Int(sqlite3_column_int64(stmt, 2))
                        mod.prepLocationId = Int(sqlite3_column_int64(stmt, 3))
                        mod.price = sqlite3_column_double(stmt, 4)
                        mod.prodcutId = Int(sqlite3_column_int64(stmt, 5))
                        mod.productName = String(cString: sqlite3_column_text(stmt, 6))
                        mod.quantity = Int(sqlite3_column_int64(stmt, 7))
                        mod.specialInstruction = String(cString: sqlite3_column_text(stmt, 8))
                        mod.subTotal = sqlite3_column_double(stmt, 9)
                        mod.updaterId = Int(sqlite3_column_int64(stmt, 10))
                        mod.groupOffline = Int(sqlite3_column_int64(stmt, 11))
                        mod.instructionPrice = sqlite3_column_double(stmt, 12)
                        mod.isItemUpdated = Int(sqlite3_column_int64(stmt, 14))
                        mod.misc = Int(sqlite3_column_int64(stmt, 15))
                        mod.sentToKitchen = Int(sqlite3_column_int64(stmt, 16)) == 1
                        mod.isDelete = Int(sqlite3_column_int64(stmt, 13)) == 1
                        mod.arrAddons = FetchOrderAddons("SELECT * FROM OrderAddons WHERE product_id = \(mod.id ?? 0)", db)
                        mod.arrIngriedent = FetchOrderIngridents("SELECT * FROM OrderIngredients WHERE product_id = \(mod.id ?? 0)", db)
                        mod.product = CoreDataHelper.shared.fetchProductObejct(prodcutId: mod.prodcutId ?? 0)
                        mod.sentToKitchenQuantity = Int(sqlite3_column_int64(stmt, 21))
                        mod.total = sqlite3_column_double(stmt, 22)
                        mod.addonPrice = sqlite3_column_double(stmt, 23)
                        mod.ingredientsPrice = sqlite3_column_double(stmt, 24)
                        result.append(mod)
                    }
                    sqlite3_finalize(stmt)
                }
            }
            sqlite3_close(db)
        }
        return result
    }
    
    static func insertOrderProducts(orderId:Int, arrProducts:[OrderProductModel], database: OpaquePointer?, isFromList:Bool = false, checkUpdated:Bool = true){
        
        let update = "UPDATE OrderProducts SET order_id = ?, order_split_id = ?, preparation_location_id = ?, price = ?, product_id = ?, product_name = ?, quantity = ?, special_instruction = ?, sub_total = ?, updater_id = ?, group_offline = ?, instruction_price = ?, isDelete = ?, isItemUpdated = ?, misc = ?, sent_to_kitchen = ?, category_id = ?, group_unique_id = ?, id = ?, updated_at = ?, sent_to_kitchen_quantity = ?, total = ?, addon_price = ?, ingrident_price = ? WHERE item_unique_id = ?"
        
        let updateAPI = "UPDATE OrderProducts SET order_id = ?, order_split_id = ?, id = ? WHERE item_unique_id = ?"
        
        let insert = "INSERT INTO OrderProducts (id, order_id, order_split_id, preparation_location_id, price, product_id, product_name, quantity, special_instruction, sub_total, updater_id, group_offline, instruction_price, isDelete, isItemUpdated, misc, sent_to_kitchen, category_id, group_unique_id, item_unique_id, updated_at, sent_to_kitchen_quantity, total, addon_price, ingrident_price) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? , ? , ?)"
        
        for product in arrProducts{
//            print(product.uniqueId)
            
            if let oldproduct = FetchOrderProducts("SELECT * FROM OrderProducts WHERE item_unique_id = '\(product.uniqueId)' LIMIT 1", database).first{
                if oldproduct.isItemUpdated == 0 || checkUpdated{
                    var statement: OpaquePointer!
                    
                    if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(product.orderSplitId ?? 0))
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(product.prepLocationId ?? 0))
                        sqlite3_bind_double(statement, 4, product.price ?? 0)
                        sqlite3_bind_int64(statement, 5, sqlite3_int64(product.prodcutId ?? 0))
                        sqlite3_bind_text(statement, 6, ((product.productName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(product.quantity ?? 0))
                        sqlite3_bind_text(statement, 8, ((product.specialInstruction ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 9, product.subTotal ?? 0)
                        if !isFromList{
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(AppConstants.userData?.id ?? 0))
                        }else{
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(product.updaterId ?? 0))
                        }
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(product.groupOffline))
                        sqlite3_bind_double(statement, 12, product.instructionPrice ?? 0)
                        sqlite3_bind_int64(statement, 13, sqlite3_int64(product.isDelete ? 1 : 0))
                        sqlite3_bind_int64(statement, 14, sqlite3_int64(product.isItemUpdated))
                        sqlite3_bind_int64(statement, 15, sqlite3_int64(product.misc ?? 0))
                        sqlite3_bind_int64(statement, 16, sqlite3_int64(product.sentToKitchen ? 1 : 0))
                        sqlite3_bind_int64(statement, 17, sqlite3_int64(product.product?.categoryId ?? 0))
                        
                        sqlite3_bind_text(statement, 18, (product.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 19, sqlite3_int64(product.id ?? 0))
                        sqlite3_bind_text(statement, 20, (product.updatedAt as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 21, sqlite3_int64(product.sentToKitchenQuantity))
                        sqlite3_bind_double(statement, 22, product.total ?? 0)
                        sqlite3_bind_double(statement, 23, product.addonPrice ?? 0)
                        sqlite3_bind_double(statement, 24, product.ingredientsPrice ?? 0)
                        sqlite3_bind_text(statement, 25, (product.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not updated product, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }else{
                    var statement: OpaquePointer!
                    
                    if sqlite3_prepare_v2(database, updateAPI, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(product.orderSplitId ?? 0))
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(product.id ?? 0))
                        sqlite3_bind_text(statement, 4, (product.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not updated product, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
            }else{
                var statement: OpaquePointer!
                
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(product.id ?? 0))
                    sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                    sqlite3_bind_int64(statement, 3, sqlite3_int64(product.orderSplitId ?? 0))
                    sqlite3_bind_int64(statement, 4, sqlite3_int64(product.prepLocationId ?? 0))
                    sqlite3_bind_double(statement, 5, product.price ?? 0)
                    sqlite3_bind_int64(statement, 6, sqlite3_int64(product.prodcutId ?? 0))
                    sqlite3_bind_text(statement, 7, ((product.productName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 8, sqlite3_int64(product.quantity ?? 0))
                    sqlite3_bind_text(statement, 9, ((product.specialInstruction ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_double(statement, 10, product.subTotal ?? 0)
                    if !isFromList{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(AppConstants.userData?.id ?? 0))
                    }else{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(product.updaterId ?? 0))
                    }
                    sqlite3_bind_int64(statement, 12, sqlite3_int64(product.groupOffline))
                    sqlite3_bind_double(statement, 13, product.instructionPrice ?? 0)
                    sqlite3_bind_int64(statement, 14, sqlite3_int64(product.isDelete ? 1 : 0))
                    sqlite3_bind_int64(statement, 15, sqlite3_int64(product.isItemUpdated))
                    sqlite3_bind_int64(statement, 16, sqlite3_int64(product.misc ?? 0))
                    sqlite3_bind_int64(statement, 17, sqlite3_int64(product.sentToKitchen ? 1 : 0))
                    sqlite3_bind_int64(statement, 18, sqlite3_int64(product.product?.categoryId ?? 0))
                    
                    sqlite3_bind_text(statement, 19, (product.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 20, (product.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 21, (product.updatedAt as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 22, sqlite3_int64(product.sentToKitchenQuantity))
                    sqlite3_bind_double(statement, 23, product.total ?? 0)
                    sqlite3_bind_double(statement, 24, product.addonPrice ?? 0)
                    sqlite3_bind_double(statement, 25, product.ingredientsPrice ?? 0)
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert product, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }
            
            _ = self.ExecuteQuery("DELETE FROM OrderAddons WHERE product_id = \(product.id ?? 0)")
            _ = self.ExecuteQuery("DELETE FROM OrderIngredients WHERE product_id = \(product.id ?? 0)")
            self.insertOrderAddons(orderId: orderId, productId: product.id ?? 0, arrAddons: product.arrAddons, database: database, isFromList: isFromList)
            self.insertOrderIngredients(orderId: orderId, productId: product.id ?? 0, arrAddons: product.arrIngriedent, database: database, isFromList: isFromList)
        }
    }
    
    //MARK: Order Addons
    static func FetchOrderAddons(_ str:String,_ db:OpaquePointer?) -> [OrderAddOnModel]{
        var result = [OrderAddOnModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
            while (sqlite3_step(stmt) == SQLITE_ROW){
                let mod = OrderAddOnModel()
                
                mod.id = Int(sqlite3_column_int64(stmt, 0))
                mod.AddonId = Int(sqlite3_column_int64(stmt, 1))
                mod.name = String(cString: sqlite3_column_text(stmt, 4))
                mod.price = sqlite3_column_double(stmt, 5)
                mod.total = sqlite3_column_double(stmt, 6)
                mod.isNewOrder = Int(sqlite3_column_int64(stmt, 7))
                mod.updaterId = Int(sqlite3_column_int64(stmt, 8))
                mod.quantity = Int(sqlite3_column_int64(stmt, 9))
                
                result.append(mod)
            }
            sqlite3_finalize(stmt)
        }
        return result
    }
    
    static func insertOrderAddons(orderId:Int, productId:Int, arrAddons:[OrderAddOnModel], database: OpaquePointer?, isFromList:Bool = false){
        
        let insert = "INSERT INTO OrderAddons (id, addon_id, product_id, order_id, addon_name, price, total, isNewOrder, updater_id, quantity) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"
//        let update = "UPDATE OrderAddons SET addon_id = ?, product_id = ?, order_id = ?, addon_name = ?, price = ?, total = ?, isNewOrder = ?, updater_id = ?, quantity = ? WHERE id = ?"
        
        for addon in arrAddons{
//            if let addon =  FetchOrderAddons("SELECT * from OrderAddons WHERE id = \(addon.id ?? 0) LIMIT 1", database).first{
//                var statement: OpaquePointer!
//                if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
//                    sqlite3_bind_int64(statement, 10, sqlite3_int64(addon.id ?? 0))
//                    sqlite3_bind_int64(statement, 1, sqlite3_int64(addon.AddonId ?? 0))
//                    sqlite3_bind_int64(statement, 2, sqlite3_int64(productId))
//                    sqlite3_bind_int64(statement, 3, sqlite3_int64(orderId))
//                    sqlite3_bind_text(statement, 4, ((addon.name ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
//                    sqlite3_bind_double(statement, 5, addon.price ?? 0)
//                    sqlite3_bind_double(statement, 6, addon.total ?? 0)
//                    sqlite3_bind_int64(statement, 7, sqlite3_int64(addon.isNewOrder))
//                    
//                    if !isFromList{
//                        sqlite3_bind_int64(statement, 8, sqlite3_int64(AppConstants.userData?.id ?? 0))
//                    }else{
//                        sqlite3_bind_int64(statement, 8, sqlite3_int64(addon.updaterId ?? 0))
//                    }
//                    sqlite3_bind_int64(statement, 9, sqlite3_int64(addon.quantity))
//                    if sqlite3_step(statement) == SQLITE_DONE {
//                    } else {
//                        let errmsg = String(cString: sqlite3_errmsg(database))
//                        print("Could not insert addon, \(errmsg)")
//                    }
//                    sqlite3_finalize(statement)
//                }
//            }else{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(addon.id ?? 0))
                    sqlite3_bind_int64(statement, 2, sqlite3_int64(addon.AddonId ?? 0))
                    sqlite3_bind_int64(statement, 3, sqlite3_int64(productId))
                    sqlite3_bind_int64(statement, 4, sqlite3_int64(orderId))
                    sqlite3_bind_text(statement, 5, ((addon.name ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_double(statement, 6, addon.price ?? 0)
                    sqlite3_bind_double(statement, 7, addon.total ?? 0)
                    sqlite3_bind_int64(statement, 8, sqlite3_int64(addon.isNewOrder))
                    if !isFromList{
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(AppConstants.userData?.id ?? 0))
                    }else{
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(addon.updaterId ?? 0))
                    }
                    sqlite3_bind_int64(statement, 10, sqlite3_int64(addon.quantity))
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert addon, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
//            }
        }
    }
    
    //MARK: Order Ingriedents
    static func FetchOrderIngridents(_ str:String,_ db:OpaquePointer?) -> [OrderIngredientModel]{
        var result = [OrderIngredientModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
            while (sqlite3_step(stmt) == SQLITE_ROW){
                let mod = OrderIngredientModel()
                
                mod.id = Int(sqlite3_column_int64(stmt, 0))
                mod.productIngredientId = Int(sqlite3_column_int64(stmt, 3))
                mod.name = String(cString: sqlite3_column_text(stmt, 4))
                mod.price = sqlite3_column_double(stmt, 5)
                mod.quantity = Int(sqlite3_column_int64(stmt, 6))
                mod.total = sqlite3_column_double(stmt, 7)
                mod.with = Int(sqlite3_column_int64(stmt, 8))
                mod.without = Int(sqlite3_column_int64(stmt, 9))
                mod.updaterId = Int(sqlite3_column_int64(stmt, 10))
                mod.isNewOrder = Int(sqlite3_column_int64(stmt, 11))
                
                result.append(mod)
            }
            sqlite3_finalize(stmt)
        }
        return result
    }
    
    static func insertOrderIngredients(orderId:Int, productId:Int, arrAddons:[OrderIngredientModel], database: OpaquePointer?,isFromList:Bool = false){
        
        let insert = "INSERT INTO OrderIngredients (id, order_id, product_id, product_ingredient_id, name, price, quantity, total, with, without, updater_id, isNewOrder) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"
//        let update = "UPDATE OrderIngredients SET order_id = ?, product_id = ?, product_ingredient_id = ?, name = ?, price = ?, quantity = ?, total = ?, with = ?, without = ?, updater_id = ?, isNewOrder = ? WHERE id = ?"
        
        for addon in arrAddons{
//            if let _ = FetchOrderIngridents("SELECT * FROM OrderIngredients WHERE id = \(addon.id ?? 0)", database).first{
//                var statement: OpaquePointer!
//                if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
//                    sqlite3_bind_int64(statement, 12, sqlite3_int64(addon.id ?? 0))
//                    sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
//                    sqlite3_bind_int64(statement, 2, sqlite3_int64(productId))
//                    sqlite3_bind_int64(statement, 3, sqlite3_int64(addon.productIngredientId ?? 0))
//                    sqlite3_bind_text(statement, 4, ((addon.name ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
//                    sqlite3_bind_double(statement, 5, addon.price ?? 0)
//                    sqlite3_bind_int64(statement, 6, sqlite3_int64(addon.quantity ?? 0))
//                    sqlite3_bind_double(statement, 7, addon.total ?? 0)
//                    sqlite3_bind_int64(statement, 8, sqlite3_int64(addon.with ?? 0))
//                    sqlite3_bind_int64(statement, 9, sqlite3_int64(addon.without ?? 0))
//                    if !isFromList{
//                        sqlite3_bind_int64(statement, 10, sqlite3_int64(AppConstants.userData?.id ?? 0))
//                    }else{
//                        sqlite3_bind_int64(statement, 10, sqlite3_int64(addon.updaterId ?? 0))
//                    }
//                    sqlite3_bind_int64(statement, 11, sqlite3_int64(addon.isNewOrder))
//                    if sqlite3_step(statement) == SQLITE_DONE {
//                    } else {
//                        let errmsg = String(cString: sqlite3_errmsg(database))
//                        print("Could not insert ingrident, \(errmsg)")
//                    }
//                    sqlite3_finalize(statement)
//                }
//            }else{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(addon.id ?? 0))
                    sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                    sqlite3_bind_int64(statement, 3, sqlite3_int64(productId))
                    sqlite3_bind_int64(statement, 4, sqlite3_int64(addon.productIngredientId ?? 0))
                    sqlite3_bind_text(statement, 5, ((addon.name ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_double(statement, 6, addon.price ?? 0)
                    sqlite3_bind_int64(statement, 7, sqlite3_int64(addon.quantity ?? 0))
                    sqlite3_bind_double(statement, 8, addon.total ?? 0)
                    sqlite3_bind_int64(statement, 9, sqlite3_int64(addon.with ?? 0))
                    sqlite3_bind_int64(statement, 10, sqlite3_int64(addon.without ?? 0))
                    if !isFromList{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(AppConstants.userData?.id ?? 0))
                    }else{
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(addon.updaterId ?? 0))
                    }
                    sqlite3_bind_int64(statement, 12, sqlite3_int64(addon.isNewOrder))
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert ingrident, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
//            }
            
        }
    }
    
    //MARK: Order Split Group
    static func FetchOrderGroups(_ str:String,_ db:OpaquePointer?, orderId:Int) -> [OrderGroupModel]{
        var result = [OrderGroupModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
            while (sqlite3_step(stmt) == SQLITE_ROW){
                let mod = OrderGroupModel()
                
                mod.id = Int(sqlite3_column_int64(stmt, 0))
                mod.groupName = String(cString: sqlite3_column_text(stmt, 2))
                mod.paidAmount = SQLiteManage.fetchSumOfTable(feild: "amount", whereString: "WHERE order_split_id = '\(mod.id ?? 0)' AND order_id = '\(orderId)'", tableName: "OrderPayment")
                mod.total = sqlite3_column_double(stmt, 4)
                mod.offline = Int(sqlite3_column_int64(stmt, 6))
                mod.uniqueId = String(cString: sqlite3_column_text(stmt, 7))
                mod.discount = sqlite3_column_double(stmt, 8)
                mod.gratuity = sqlite3_column_double(stmt, 9)
                mod.serviceCharge = sqlite3_column_double(stmt, 10)
                mod.autoGratuityRemove = Int(sqlite3_column_int64(stmt, 11)) == 1
                mod.autoDiscountRemove = Int(sqlite3_column_int64(stmt, 12)) == 1
                mod.autoServiceChargeRemove = Int(sqlite3_column_int64(stmt, 13)) == 1
                mod.subTotal = sqlite3_column_double(stmt, 14)
                
                result.append(mod)
            }
            sqlite3_finalize(stmt)
        }
        return result
    }
    
    static func fetchOrderGroupData(orderId:Int)->[OrderGroupModel]{
        var arrGroup = [OrderGroupModel]()
        
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        
        if (sqlite3_open(DBpath, &db)==SQLITE_OK){
            arrGroup = FetchOrderGroups("SELECT * FROM OrderGroup WHERE order_id = \(orderId)", db, orderId: orderId)
        }
        
        sqlite3_close(db)
        
        return arrGroup
    }
    
    static func fetchOfflineOrderGroupData(orderId:Int)->OrderGroupModel?{
        var group:OrderGroupModel?
        
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        
        if (sqlite3_open(DBpath, &db)==SQLITE_OK){
            group = FetchOrderGroups("SELECT * FROM OrderGroup WHERE order_id = \(orderId) AND is_updated = 1 LIMIT 1", db, orderId: orderId).first
        }
        
        sqlite3_close(db)
        
        return group
    }
    
    static func insertOrderGroups(orderId:Int, arrGroups:[OrderGroupModel], database: OpaquePointer?, updated: Int = 0, offline:Int = 0, updateOffline:Bool = true){
        
        var update = "UPDATE OrderGroup SET order_id = ?, group_name = ?, paid_amount = ?, total = ?, is_updated = ?, discount = ?, gratuity = ?, service_charge = ?, auto_gratuity_remove = ?, auto_discount_remove = ?, auto_service_charge_remove = ?, sub_total = ?, id = ?, offline = ? WHERE unique_id = ?"
        if !updateOffline{
            update = "UPDATE OrderGroup SET order_id = ?, group_name = ?, paid_amount = ?, total = ?, is_updated = ?, discount = ?, gratuity = ?, service_charge = ?, auto_gratuity_remove = ?, auto_discount_remove = ?, auto_service_charge_remove = ?, sub_total = ?, id = ? WHERE unique_id = ?"
        }
        let insert = "INSERT INTO OrderGroup (id, order_id, group_name, paid_amount, total, is_updated, offline, unique_id, discount, gratuity, service_charge, auto_gratuity_remove, auto_discount_remove, auto_service_charge_remove, sub_total) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"
        
        if database == nil{
            let dbPath = copyDatabaseToDirectory()
            
            var database: OpaquePointer? = nil
            
            if (sqlite3_open(dbPath, &database)==SQLITE_OK)
            {
                for group in arrGroups{
                    if let _ = FetchOrderGroups("SELECT * FROM OrderGroup WHERE unique_id = \(group.uniqueId)", database, orderId: orderId).first{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                            sqlite3_bind_text(statement, 2, ((group.groupName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 3, (group.paidAmount ?? 0).getRoundFigure())
                            sqlite3_bind_double(statement, 4, (group.total ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 5, sqlite3_int64(updated))
                            sqlite3_bind_double(statement, 6, group.discount.getRoundFigure())
                            sqlite3_bind_double(statement, 7, group.gratuity.getRoundFigure())
                            sqlite3_bind_double(statement, 8, group.serviceCharge.getRoundFigure())
                            sqlite3_bind_int64(statement, 9, sqlite3_int64(group.autoGratuityRemove ? 1 : 0))
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(group.autoDiscountRemove ? 1 : 0))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(group.autoServiceChargeRemove ? 1 : 0))
                            sqlite3_bind_double(statement, 12, group.subTotal.getRoundFigure())
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(group.id ?? 0))
                            if updateOffline{
                                sqlite3_bind_int64(statement, 14, sqlite3_int64(offline))
                                sqlite3_bind_text(statement, 15, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            }else{
                                sqlite3_bind_text(statement, 14, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            }
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not updated group, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }else{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(group.id ?? 0))
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                            sqlite3_bind_text(statement, 3, ((group.groupName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 4, (group.paidAmount ?? 0).getRoundFigure())
                            sqlite3_bind_double(statement, 5, (group.total ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(updated))
                            sqlite3_bind_int64(statement, 7, sqlite3_int64(offline))
                            sqlite3_bind_text(statement, 8, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 9, group.discount.getRoundFigure())
                            sqlite3_bind_double(statement, 10, group.gratuity.getRoundFigure())
                            sqlite3_bind_double(statement, 11, group.serviceCharge.getRoundFigure())
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(group.autoGratuityRemove ? 1 : 0))
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(group.autoDiscountRemove ? 1 : 0))
                            sqlite3_bind_int64(statement, 14, sqlite3_int64(group.autoServiceChargeRemove ? 1 : 0))
                            sqlite3_bind_double(statement, 15, group.subTotal.getRoundFigure())
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not insert group, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }
            }
            sqlite3_close(database)
        }else{
            for group in arrGroups{
                if let _ = FetchOrderGroups("SELECT * FROM OrderGroup WHERE unique_id = \(group.uniqueId)", database, orderId: orderId).first{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                        sqlite3_bind_text(statement, 2, ((group.groupName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 3, (group.paidAmount ?? 0).getRoundFigure())
                        sqlite3_bind_double(statement, 4, (group.total ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 5, sqlite3_int64(updated))
                        sqlite3_bind_double(statement, 6, group.discount.getRoundFigure())
                        sqlite3_bind_double(statement, 7, group.gratuity.getRoundFigure())
                        sqlite3_bind_double(statement, 8, group.serviceCharge.getRoundFigure())
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(group.autoGratuityRemove ? 1 : 0))
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(group.autoDiscountRemove ? 1 : 0))
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(group.autoServiceChargeRemove ? 1 : 0))
                        sqlite3_bind_double(statement, 12, group.subTotal.getRoundFigure())
                        sqlite3_bind_int64(statement, 13, sqlite3_int64(group.id ?? 0))
                        if updateOffline{
                            sqlite3_bind_int64(statement, 14, sqlite3_int64(offline))
                            sqlite3_bind_text(statement, 15, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        }else{
                            sqlite3_bind_text(statement, 14, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        }
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not updated group, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }else{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(group.id ?? 0))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                        sqlite3_bind_text(statement, 3, ((group.groupName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 4, (group.paidAmount ?? 0).getRoundFigure())
                        sqlite3_bind_double(statement, 5, (group.total ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(updated))
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(offline))
                        sqlite3_bind_text(statement, 8, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 9, group.discount.getRoundFigure())
                        sqlite3_bind_double(statement, 10, group.gratuity.getRoundFigure())
                        sqlite3_bind_double(statement, 11, group.serviceCharge.getRoundFigure())
                        sqlite3_bind_int64(statement, 12, sqlite3_int64(group.autoGratuityRemove ? 1 : 0))
                        sqlite3_bind_int64(statement, 13, sqlite3_int64(group.autoDiscountRemove ? 1 : 0))
                        sqlite3_bind_int64(statement, 14, sqlite3_int64(group.autoServiceChargeRemove ? 1 : 0))
                        sqlite3_bind_double(statement, 15, group.subTotal.getRoundFigure())
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not insert group, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
            }
        }
    }
    
    static func updateOrderGroupTotal(arrGroups:[OrderGroupModel]){
        
        let update = "UPDATE OrderGroup SET total = ?, is_updated = ?, discount = ?, gratuity = ?, service_charge = ?, auto_gratuity_remove = ?, auto_discount_remove = ?, auto_service_charge_remove = ? WHERE unique_id = ?"
        
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK)
        {
            for group in arrGroups{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_double(statement, 1, (group.total ?? 0).getRoundFigure())
                    sqlite3_bind_int64(statement, 2, 1)
                    sqlite3_bind_double(statement, 3, group.discount)
                    sqlite3_bind_double(statement, 4, group.gratuity)
                    sqlite3_bind_double(statement, 5, group.serviceCharge)
                    sqlite3_bind_int64(statement, 6, group.autoGratuityRemove ? 1 : 0)
                    sqlite3_bind_int64(statement, 7, group.autoDiscountRemove ? 1 : 0)
                    sqlite3_bind_int64(statement, 8, group.autoServiceChargeRemove ? 1 : 0)
                    sqlite3_bind_text(statement, 9, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not updated group, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }
        }
        sqlite3_close(database)
    }
    
    static func updateOrderGroupTotal(group:OrderGroupModel){
        
        let update = "UPDATE OrderGroup SET total = ?, is_updated = ?, discount = ?, gratuity = ?, service_charge = ?, auto_gratuity_remove = ?, auto_discount_remove = ?, auto_service_charge_remove = ? WHERE unique_id = ?"
        
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK)
        {
            var statement: OpaquePointer!
            if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                sqlite3_bind_double(statement, 1, (group.total ?? 0).getRoundFigure())
                sqlite3_bind_int64(statement, 2, 1)
                sqlite3_bind_double(statement, 3, group.discount)
                sqlite3_bind_double(statement, 4, group.gratuity)
                sqlite3_bind_double(statement, 5, group.serviceCharge)
                sqlite3_bind_int64(statement, 6, group.autoGratuityRemove ? 1 : 0)
                sqlite3_bind_int64(statement, 7, group.autoDiscountRemove ? 1 : 0)
                sqlite3_bind_int64(statement, 8, group.autoServiceChargeRemove ? 1 : 0)
                sqlite3_bind_text(statement, 9, (group.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                if sqlite3_step(statement) == SQLITE_DONE {
                } else {
                    let errmsg = String(cString: sqlite3_errmsg(database))
                    print("Could not updated group, \(errmsg)")
                }
                sqlite3_finalize(statement)
            }
        }
        sqlite3_close(database)
    }
    
    //MARK: Order Payments
    static func FetchOrderPayments(_ str:String,_ db:OpaquePointer?) -> [PaymentOrderModel]{
        var result = [PaymentOrderModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
            while (sqlite3_step(stmt) == SQLITE_ROW){
                let mod = PaymentOrderModel()
                
                mod.id = Int(sqlite3_column_int64(stmt, 0))
                mod.orderSplitId = Int(sqlite3_column_int64(stmt, 2))
                mod.paymentMethodName = String(cString: sqlite3_column_text(stmt, 3))
                mod.amount = sqlite3_column_double(stmt, 4)
                mod.offline = Int(sqlite3_column_int64(stmt, 5))
                mod.voucherId = Int(sqlite3_column_int64(stmt, 6))
                mod.paymentMethodId = Int(sqlite3_column_int64(stmt, 8))
                mod.chequeNumber = String(cString: sqlite3_column_text(stmt, 9))
                mod.ccNumber = String(cString: sqlite3_column_text(stmt, 10))
                mod.txnId = String(cString: sqlite3_column_text(stmt, 11))
                mod.updaterId = Int(sqlite3_column_int64(stmt, 12))
                mod.groupUniqueId = String(cString: sqlite3_column_text(stmt, 13))
                mod.isRefunded = Int(sqlite3_column_int64(stmt, 14))
                //"\(Int(sqlite3_column_int64(stmt, 13)))"

                result.append(mod)
            }
            sqlite3_finalize(stmt)
        }
        return result
    }
    
    static func fetchOrderPaymentData(orderId:Int, groupId:Int? = nil,offline:Bool = false)->[PaymentOrderModel]{
        var arrPayment = [PaymentOrderModel]()
        
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        
        if groupId != nil{
            
            if (sqlite3_open(DBpath, &db)==SQLITE_OK){
                if offline{
                    arrPayment = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_split_id = \(groupId!) AND offline = 1", db)
                }else{
                    arrPayment = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_split_id = \(groupId!)", db)
                }
            }
            sqlite3_close(db)
            
        }else{
            
            if (sqlite3_open(DBpath, &db)==SQLITE_OK){
                if offline{
                    arrPayment = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_id = \(orderId) AND offline = 1", db)
                }else{
                    arrPayment = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_id = \(orderId)", db)
                }
            }
            sqlite3_close(db)
        }
        return arrPayment
    }
    
    static func fetchOfflineOrderPaymentData(orderId:Int)->PaymentOrderModel?{
        var group:PaymentOrderModel?
        
        let DBpath:String = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        
        if (sqlite3_open(DBpath, &db)==SQLITE_OK){
            group = FetchOrderPayments("SELECT * FROM OrderPayment WHERE order_id = \(orderId) AND offline = 1 LIMIT 1", db).first
        }
        
        sqlite3_close(db)
        
        return group
    }
    
    static func insertOrderPayments(orderId:Int, arrPayments:[PaymentOrderModel], database: OpaquePointer?, voucherCode:String = "", offline:Int = 0, isFromList:Bool = false){
        
        let update = "UPDATE OrderPayment SET order_id = ?, order_split_id = ?, payment_method_name = ?, amount = ?, offline = ?, voucher_id = ?, voucher_code = ?, payment_method_id = ?, cheque_number = ?, cc_number = ?, id = ?, updater_id = ?, group_unique_id = ?, is_refunded = ? WHERE txn_id = ?"
        let insert = "INSERT INTO OrderPayment (id, order_id, order_split_id, payment_method_name, amount, offline, voucher_id, voucher_code, payment_method_id, cheque_number, cc_number, txn_id, updater_id, group_unique_id, is_refunded) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        
        if database != nil{
            for payment in arrPayments{
                if let _ = FetchOrderPayments("SELECT * FROM OrderPayment WHERE txn_id = '\(payment.txnId ?? "")'", database).first{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(payment.orderSplitId ?? 0))
                        sqlite3_bind_text(statement, 3, ((payment.paymentMethodName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 4, (payment.amount ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 5, sqlite3_int64(offline))
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(payment.voucherId ?? 0))
                        sqlite3_bind_text(statement, 7, (voucherCode as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 8, sqlite3_int64(payment.paymentMethodId ?? 0))
                        sqlite3_bind_text(statement, 9, ((payment.chequeNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 10, ((payment.ccNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(payment.id ?? 0))
                        if !isFromList{
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(AppConstants.userData?.id ?? 0))
                        }else{
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(payment.updaterId ?? 0))
                        }
                        sqlite3_bind_text(statement, 13, (payment.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 14, sqlite3_int64(payment.isRefunded))
                        sqlite3_bind_text(statement, 15, ((payment.txnId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not updated payment, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }else{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(payment.id ?? 0))
                        sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(payment.orderSplitId ?? 0))
                        sqlite3_bind_text(statement, 4, ((payment.paymentMethodName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 5, (payment.amount ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 6, sqlite3_int64(offline))
                        sqlite3_bind_int64(statement, 7, sqlite3_int64(payment.voucherId ?? 0))
                        if voucherCode != ""{
                            sqlite3_bind_text(statement, 8, (voucherCode as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        }
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(payment.paymentMethodId ?? 0))
                        sqlite3_bind_text(statement, 10, ((payment.chequeNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 11, ((payment.ccNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 12, ((payment.txnId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        if !isFromList{
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(AppConstants.userData?.id ?? 0))
                        }else{
                            sqlite3_bind_int64(statement, 13, sqlite3_int64(payment.updaterId ?? 0))
                        }
                        sqlite3_bind_text(statement, 14, (payment.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_int64(statement, 15, sqlite3_int64(payment.isRefunded))
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not insert payment, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
                
            }
        }else{
            let dbPath = copyDatabaseToDirectory()
            
            var database: OpaquePointer? = nil
            
            if (sqlite3_open(dbPath, &database)==SQLITE_OK)
            {
                for payment in arrPayments{
                    if let _ = FetchOrderPayments("SELECT * FROM OrderPayment WHERE id = \(payment.id ?? 0)", database).first{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(orderId))
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(payment.orderSplitId ?? 0))
                            sqlite3_bind_text(statement, 3, ((payment.paymentMethodName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 4, (payment.amount ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 5, sqlite3_int64(offline))
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(payment.voucherId ?? 0))
                            sqlite3_bind_text(statement, 7, (voucherCode as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 8, sqlite3_int64(payment.paymentMethodId ?? 0))
                            sqlite3_bind_text(statement, 9, ((payment.chequeNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 10, ((payment.ccNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(payment.id ?? 0))
                            if !isFromList{
                                sqlite3_bind_int64(statement, 12, sqlite3_int64(AppConstants.userData?.id ?? 0))
                            }else{
                                sqlite3_bind_int64(statement, 12, sqlite3_int64(payment.updaterId ?? 0))
                            }
                            sqlite3_bind_text(statement, 13, (payment.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 14, sqlite3_int64(payment.isRefunded))
                            sqlite3_bind_text(statement, 15, ((payment.txnId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not insert payment, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }else{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(payment.id ?? 0))
                            sqlite3_bind_int64(statement, 2, sqlite3_int64(orderId))
                            sqlite3_bind_int64(statement, 3, sqlite3_int64(payment.orderSplitId ?? 0))
                            sqlite3_bind_text(statement, 4, ((payment.paymentMethodName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 5, (payment.amount ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 6, sqlite3_int64(offline))
                            sqlite3_bind_int64(statement, 7, sqlite3_int64(payment.voucherId ?? 0))
                            if voucherCode != ""{
                                sqlite3_bind_text(statement, 8, (voucherCode as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            }
                            sqlite3_bind_int64(statement, 9, sqlite3_int64(payment.paymentMethodId ?? 0))
                            sqlite3_bind_text(statement, 10, ((payment.chequeNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 11, ((payment.ccNumber ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 12, ((payment.txnId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            if !isFromList{
                                sqlite3_bind_int64(statement, 13, sqlite3_int64(AppConstants.userData?.id ?? 0))
                            }else{
                                sqlite3_bind_int64(statement, 13, sqlite3_int64(payment.updaterId ?? 0))
                            }
                            sqlite3_bind_text(statement, 14, (payment.groupUniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_int64(statement, 15, sqlite3_int64(payment.isRefunded))
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not insert payment, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }
            }
            sqlite3_close(database)
        }
        
    }
    
    static func updateOrderTotalPaid(id:String, amount:Double, groupId:String? = nil, orderTotal:Double = 0){
        if groupId != nil{
            _ = ExecuteQuery("UPDATE OrderGroup SET paid_amount = \(amount) WHERE unique_id = \(groupId!)")
            _ = ExecuteQuery("UPDATE Orders SET total_paid = \(orderTotal) WHERE unique_id = \(id)")
        }else{
            _ = ExecuteQuery("UPDATE Orders SET total_paid = \(amount) WHERE unique_id = \(id)")
        }
    }
    
    //MARK: Voucher Operation
    static func insertVouchers(arrVouchers:[VoucherModel], database: OpaquePointer? = nil){
        let update = "UPDATE Vouchers SET voucher_code = ?, voucher_amount = ?, customer_id = ?, order_type_id = ?, start_date = ?, end_date = ?, week_day = ?, minimum_order_value = ?, applied = ?, offline = ?, updated = ?, id = ? WHERE unique_id = ?"
        let insert = "INSERT INTO Vouchers (id, voucher_code, voucher_amount, customer_id, order_type_id, start_date, end_date, week_day, minimum_order_value, applied, offline, updated, unique_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        
        if database != nil{
            for voucher in arrVouchers{
                if voucher.customer != nil{
                    CoreDataHelper.shared.saveAndUpdateCusomer(customer: voucher.customer!, offline: 0, updated: 0)
                }
                if let oldVoucher = FetchVouchers("SELECT * FROM Vouchers WHERE unique_id = \(voucher.uniqueId)", database).first{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                        sqlite3_bind_text(statement, 1, ((voucher.voucherCode ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 2, (voucher.voucherAmount ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 3, sqlite3_int64(voucher.customerId ?? 0))
                        sqlite3_bind_text(statement, 4, ((voucher.orderTypeId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 5, ((voucher.startDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 6, ((voucher.endDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 7, ((voucher.weekDay ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 8, (voucher.minimumOrderValue ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 9, sqlite3_int64(voucher.applied ?? 0))
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(voucher.offline))
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(voucher.updated))
                        sqlite3_bind_int64(statement, 12, sqlite3_int64(voucher.id ?? 0))
//                        sqlite3_bind_int64(statement, 13, sqlite3_int64(voucher.uniqueId))
                        sqlite3_bind_text(statement, 13, (voucher.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not updated voucher, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }else{
                    var statement: OpaquePointer!
                    if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                        
                        sqlite3_bind_int64(statement, 1, sqlite3_int64(voucher.id ?? 0))
                        sqlite3_bind_text(statement, 2, ((voucher.voucherCode ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 3, (voucher.voucherAmount ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 4, sqlite3_int64(voucher.customerId ?? 0))
                        sqlite3_bind_text(statement, 5, ((voucher.orderTypeId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 6, ((voucher.startDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 7, ((voucher.endDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_text(statement, 8, ((voucher.weekDay ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        sqlite3_bind_double(statement, 9, (voucher.minimumOrderValue ?? 0).getRoundFigure())
                        sqlite3_bind_int64(statement, 10, sqlite3_int64(voucher.applied ?? 0))
                        sqlite3_bind_int64(statement, 11, sqlite3_int64(voucher.offline))
                        sqlite3_bind_int64(statement, 12, sqlite3_int64(voucher.updated))
//                        sqlite3_bind_int64(statement, 13, sqlite3_int64(voucher.uniqueId))
                        sqlite3_bind_text(statement, 13, (voucher.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                        if sqlite3_step(statement) == SQLITE_DONE {
                        } else {
                            let errmsg = String(cString: sqlite3_errmsg(database))
                            print("Could not insert voucher, \(errmsg)")
                        }
                        sqlite3_finalize(statement)
                    }
                }
            }
        }else{
            let dbPath = copyDatabaseToDirectory()
            
            var database: OpaquePointer? = nil
            
            if (sqlite3_open(dbPath, &database)==SQLITE_OK)
            {
                for voucher in arrVouchers{
                    if let _ = FetchVouchers("SELECT * FROM Vouchers WHERE unique_id = \(voucher.uniqueId)", database).first{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, update, -1, &statement, nil) == SQLITE_OK {
                            sqlite3_bind_text(statement, 1, ((voucher.voucherCode ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 2, (voucher.voucherAmount ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 3, sqlite3_int64(voucher.customerId ?? 0))
                            sqlite3_bind_text(statement, 4, ((voucher.orderTypeId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 5, ((voucher.startDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 6, ((voucher.endDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 7, ((voucher.weekDay ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 8, (voucher.minimumOrderValue ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 9, sqlite3_int64(voucher.applied ?? 0))
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(voucher.offline))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(voucher.updated))
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(voucher.id ?? 0))
//                            sqlite3_bind_int64(statement, 13, sqlite3_int64(voucher.uniqueId))
                            sqlite3_bind_text(statement, 13, (voucher.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not updated voucher, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }else{
                        var statement: OpaquePointer!
                        if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                            
                            sqlite3_bind_int64(statement, 1, sqlite3_int64(voucher.id ?? 0))
                            sqlite3_bind_text(statement, 2, ((voucher.voucherCode ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 3, (voucher.voucherAmount ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 4, sqlite3_int64(voucher.customerId ?? 0))
                            sqlite3_bind_text(statement, 5, ((voucher.orderTypeId ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 6, ((voucher.startDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 7, ((voucher.endDate ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_text(statement, 8, ((voucher.weekDay ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            sqlite3_bind_double(statement, 9, (voucher.minimumOrderValue ?? 0).getRoundFigure())
                            sqlite3_bind_int64(statement, 10, sqlite3_int64(voucher.applied ?? 0))
                            sqlite3_bind_int64(statement, 11, sqlite3_int64(voucher.offline))
                            sqlite3_bind_int64(statement, 12, sqlite3_int64(voucher.updated))
//                            sqlite3_bind_int64(statement, 13, sqlite3_int64(voucher.uniqueId))
                            sqlite3_bind_text(statement, 13, (voucher.uniqueId as NSString).utf8String, -1, SQLITE_TRANSIENT)
                            if sqlite3_step(statement) == SQLITE_DONE {
                            } else {
                                let errmsg = String(cString: sqlite3_errmsg(database))
                                print("Could not insert voucher, \(errmsg)")
                            }
                            sqlite3_finalize(statement)
                        }
                    }
                }
            }
            sqlite3_close(database)
        }
    }
    
    static func FetchUpdatedVouchers() -> [VoucherModel]{
        return FetchVouchers("Select * FROM Vouchers WHERE updated == 1")
    }
    
    static func FetchVouchers(_ str:String, _ db:OpaquePointer? = nil) -> [VoucherModel]{
        
        var result = [VoucherModel]()
        var stmt:OpaquePointer? = nil
        
        let strExec=str.cString(using: String.Encoding.utf8)
        
        if db != nil{
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = VoucherModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.voucherCode = String(cString: sqlite3_column_text(stmt, 1))
                    mod.voucherAmount = sqlite3_column_double(stmt, 2)
                    mod.customerId = Int(sqlite3_column_int64(stmt, 3))
                    mod.orderTypeId = String(cString: sqlite3_column_text(stmt, 4))
                    mod.startDate = String(cString: sqlite3_column_text(stmt, 5))
                    mod.endDate = String(cString: sqlite3_column_text(stmt, 6))
                    mod.weekDay = String(cString: sqlite3_column_text(stmt, 7))
                    mod.minimumOrderValue = sqlite3_column_double(stmt, 8)
                    mod.applied = Int(sqlite3_column_int64(stmt, 9))
                    mod.offline = Int(sqlite3_column_int64(stmt, 10))
                    mod.updated = Int(sqlite3_column_int64(stmt, 11))
                    mod.customer = CoreDataHelper.shared.fetchCustomerObect(id: mod.customerId ?? 0)
                    mod.uniqueId = String(cString: sqlite3_column_text(stmt, 12))
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }else{
            
            let DBpath:String = copyDatabaseToDirectory()
            
            var db: OpaquePointer? = nil
            if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
                if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                    while (sqlite3_step(stmt) == SQLITE_ROW){
                        let mod = VoucherModel()
                        
                        mod.id = Int(sqlite3_column_int64(stmt, 0))
                        mod.voucherCode = String(cString: sqlite3_column_text(stmt, 1))
                        mod.voucherAmount = sqlite3_column_double(stmt, 2)
                        mod.customerId = Int(sqlite3_column_int64(stmt, 3))
                        mod.orderTypeId = String(cString: sqlite3_column_text(stmt, 4))
                        mod.startDate = String(cString: sqlite3_column_text(stmt, 5))
                        mod.endDate = String(cString: sqlite3_column_text(stmt, 6))
                        mod.weekDay = String(cString: sqlite3_column_text(stmt, 7))
                        mod.minimumOrderValue = sqlite3_column_double(stmt, 8)
                        mod.applied = Int(sqlite3_column_int64(stmt, 9))
                        mod.offline = Int(sqlite3_column_int64(stmt, 10))
                        mod.updated = Int(sqlite3_column_int64(stmt, 11))
                        mod.customer = CoreDataHelper.shared.fetchCustomerObect(id: mod.customerId ?? 0)
                        mod.uniqueId = String(cString: sqlite3_column_text(stmt, 12))
                        result.append(mod)
                    }
                    sqlite3_finalize(stmt)
                }
            }
            
            sqlite3_close(db)
        }
        return result
    }
    
    //MARK: Utility Database Operation
    static func fetchOrderStatuseById(id:Int)->OrderStatusModel?{
        var stmt:OpaquePointer? = nil

        let strExec="SELECT * FROM OrderStatus WHERE id = \(id) LIMIT 1".cString(using: String.Encoding.utf8)

        let DBpath = copyDatabaseToDirectory()

        var db: OpaquePointer? = nil
        var mod:OrderStatusModel?
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    mod = OrderStatusModel()
                    mod!.id = Int(sqlite3_column_int64(stmt, 0))
                    mod!.status = String(cString: sqlite3_column_text(stmt, 1))
                    mod!.topColor = String(cString: sqlite3_column_text(stmt, 2))
                    mod!.bottomColor = String(cString: sqlite3_column_text(stmt, 3))
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        return mod
    }
    
    static func fetchOrderStatusesData()->[OrderStatusModel]{
        var result = [OrderStatusModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec="SELECT * FROM OrderStatus".cString(using: String.Encoding.utf8)
        
        let DBpath = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = OrderStatusModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.status = String(cString: sqlite3_column_text(stmt, 1))
                    mod.topColor = String(cString: sqlite3_column_text(stmt, 2))
                    mod.bottomColor = String(cString: sqlite3_column_text(stmt, 3))
                    
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        return result
    }
    
    static func saveOrderStatusIntoDatabase(arrData:[OrderStatusModel]){
        
        let insert = "INSERT INTO OrderStatus (id, status, top_color, bottom_color) VALUES (?, ?, ? ,?)"
        
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK){
            _ = ExecuteQuery("DELETE FROM OrderStatus", database)
            for data in arrData{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(data.id ?? 0))
                    sqlite3_bind_text(statement, 2, ((data.status ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 3, ((data.topColor ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 4, ((data.bottomColor ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert order status, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }else{
                    
                }
            }
        }
        sqlite3_close(database)
    }
    
    
    static func fetchOrderTypesData()->[OrderTypesModel]{
        var result = [OrderTypesModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec="SELECT * FROM OrderTypes".cString(using: String.Encoding.utf8)
        
        let DBpath = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = OrderTypesModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.type = String(cString: sqlite3_column_text(stmt, 1))
                    mod.topColor = String(cString: sqlite3_column_text(stmt, 2))
                    mod.bottomColor = String(cString: sqlite3_column_text(stmt, 3))
                    
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }
        sqlite3_close(db)
        
        return result
    }
    
    static func saveOrderTypeIntoDatabase(arrData:[OrderTypesModel]){
        
        let insert = "INSERT INTO OrderTypes (id, type, top_color, bottom_color) VALUES (?, ?, ?, ?)"
        
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK){
            _ = ExecuteQuery("DELETE FROM OrderTypes", database)
            for data in arrData{
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(data.id ?? 0))
                    sqlite3_bind_text(statement, 2, ((data.type ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 3, ((data.topColor ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_text(statement, 4, ((data.bottomColor ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert order type, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }
        }
        sqlite3_close(database)
    }
    
    static func fetchPatmentMethods()->[PaymentMethodModel]{
        var result = [PaymentMethodModel]()
        
        var stmt:OpaquePointer? = nil
        
        let strExec="SELECT * FROM PaymentMethods WHERE disabled == 0".cString(using: String.Encoding.utf8)
        
        let DBpath = copyDatabaseToDirectory()
        
        var db: OpaquePointer? = nil
        if ( sqlite3_open(DBpath,&db) == SQLITE_OK){
            if (sqlite3_prepare_v2(db, strExec! , -1, &stmt, nil) == SQLITE_OK){
                while (sqlite3_step(stmt) == SQLITE_ROW){
                    let mod = PaymentMethodModel()
                    
                    mod.id = Int(sqlite3_column_int64(stmt, 0))
                    mod.paymentMethodName = String(cString: sqlite3_column_text(stmt, 1))
                    
                    result.append(mod)
                }
                sqlite3_finalize(stmt)
            }
        }
        
        sqlite3_close(db)
        return result
    }
    
    static func savePaymentMethodIntoDatabase(arrData:[PaymentMethodModel]){
        
        let insert = "INSERT INTO PaymentMethods (id, name, disabled) VALUES (?, ?, ?)"
        
        let dbPath = copyDatabaseToDirectory()
        
        var database: OpaquePointer? = nil
        
        if (sqlite3_open(dbPath, &database)==SQLITE_OK){
            _ = ExecuteQuery("DELETE FROM PaymentMethods", database)
            for i in 0..<arrData.count{
                AppCommonMethods.changeProgressMessage(msg: "Saving payment methods\n\(i+1)/\(arrData.count)")
                let data = arrData[i]
                var statement: OpaquePointer!
                if sqlite3_prepare_v2(database, insert, -1, &statement, nil) == SQLITE_OK {
                    sqlite3_bind_int64(statement, 1, sqlite3_int64(data.id ?? 0))
                    sqlite3_bind_text(statement, 2, ((data.paymentMethodName ?? "") as NSString).utf8String, -1, SQLITE_TRANSIENT)
                    sqlite3_bind_int64(statement, 3, sqlite3_int64(data.disabled ?? 0))
                    
                    if sqlite3_step(statement) == SQLITE_DONE {
                    } else {
                        let errmsg = String(cString: sqlite3_errmsg(database))
                        print("Could not insert order payment, \(errmsg)")
                    }
                    sqlite3_finalize(statement)
                }
            }
        }
        sqlite3_close(database)
    }
    
}
