//
//  Validation.swift
//  TiffinTom
//
//  Created by Tushar Premal on 10/07/20.
//  Copyright © 2020 Tushar Premal. All rights reserved.
//

import Foundation
import UIKit

enum ValidationType {
    case email
    case stringWithFirstLetterCaps
    case phoneNo
    case alphabeticString
    case password
    case emptyField
}

enum RegEx: String {
    case email = "[A-Z0-9a-z._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}" // Email
    case password = "(?=.*[a-z])(?=.*[0-9])(?=.*[A-Za-z]).{8,16}" // Password length 6-15
    case alphabeticStringWithSpace = "^[a-zA-Z ]*$" // e.g. hello sandeep
    case alphabeticStringFirstLetterCaps = "^[A-Z]+[a-zA-Z]*$" // SandsHell
    case phoneNo = "[0-9]{10,14}" // PhoneNo 10-14 Digits
    
    //Change RegEx according to your Requirement
}

enum BottomAlertType{
    case kNormal
    case kOrder
    case kReservation
}

struct BottomAlertSettings{
    var strMessage = ""
    var isClickable = false
    var id = 0
    var orderType = ""
    var type = BottomAlertType.kNormal
    var timeToHide = 5
    var bgColor:UIColor?
    var reservationModel:OnlineReservationModel!
}

class AppValidation: NSObject,AlertControllerDelegate {
    
    static let shared = AppValidation()
    var settings:BottomAlertSettings!
    var bottomAlert:UIView?
    var handler:((Int) -> Void )!
    var isPresented = false
    
    //MARK: - BottomAlertView
    func showBottomAlertView(settings:BottomAlertSettings){
        if settings.strMessage == ""{
            return
        }
        if bottomAlert != nil{
            self.bottomAlert!.removeFromSuperview()
            self.bottomAlert = nil
        }
        self.settings = settings
        var height = settings.strMessage.getHeightFromString(font: AppConstants.GlobalFontConstants.kRegularFont(size: 17), width: AppConstants.ScreenSize.SCREEN_WIDTH-32)
        height = max(40, (height + 16))
        let width = AppConstants.ScreenSize.SCREEN_WIDTH
        bottomAlert = UIView.init(frame: CGRect.init(x: 0, y: AppConstants.ScreenSize.SCREEN_HEIGHT + 30 , width: width, height: height + AppConstants.ScreenSize.bottomPadding))
        bottomAlert!.backgroundColor = AppConstants.Colors.kAppDarkGrayColor
        if settings.bgColor != nil{
            bottomAlert!.backgroundColor = settings.bgColor!
        }
        let lblMessage = UILabel.init(frame: CGRect.init(x: 8, y: 8, width: width - 16, height: height - 16))
        lblMessage.textColor = .white
        lblMessage.numberOfLines = 0
        lblMessage.text = settings.strMessage
        lblMessage.font = AppConstants.GlobalFontConstants.kRegularFont(size: 17)
        bottomAlert!.addSubview(lblMessage)
        
        if settings.isClickable{
            lblMessage.frame = CGRect.init(x: 8, y: 8, width: width - 70, height: height - 16)
            let btnView = UIButton.init(frame: CGRect.init(x: width-70, y: 0, width: 70, height: height))
            btnView.addTarget(self, action: #selector(btnViewAction), for: .touchUpInside)
            let strView = NSAttributedString.init(string: "View", attributes: [NSAttributedString.Key.underlineStyle : NSUnderlineStyle.single.rawValue,NSAttributedString.Key.foregroundColor:UIColor.white,NSAttributedString.Key.font:AppConstants.GlobalFontConstants.kRegularFont(size: 14)])
            btnView.setAttributedTitle(strView, for: .normal)
            bottomAlert!.addSubview(btnView)
        }
        
        AppConstants.appDelegate.window!.addSubview(bottomAlert!)
        
        UIView.animate(withDuration: 0.5) {
            self.bottomAlert!.frame.origin.y = AppConstants.ScreenSize.SCREEN_HEIGHT - AppConstants.ScreenSize.bottomPadding - height
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + DispatchTimeInterval.seconds(settings.timeToHide)) {
            if self.bottomAlert != nil{
                UIView.animate(withDuration: 0.5, animations: {
                    self.bottomAlert!.frame.origin.y = AppConstants.ScreenSize.SCREEN_HEIGHT + 30
                }) { (success) in
                    self.bottomAlert!.removeFromSuperview()
                    self.bottomAlert = nil
                }
            }
        }
    }
    
    @IBAction func btnViewAction(){
        UIView.animate(withDuration: 0.5, animations: {
            self.bottomAlert!.frame.origin.y = AppConstants.ScreenSize.SCREEN_HEIGHT + 30
        }) { (success) in
            self.bottomAlert!.removeFromSuperview()
            self.bottomAlert = nil
        }
        if settings.type == .kOrder{
            if let topVC = UIApplication.topViewController() as? SideMenuVC{
                if let VC = OrderDetailVC.addToParentView(parentVC: topVC, orderId: settings.id, orderType: settings.orderType){
                    if let controller = topVC.children.last as? OrderListVC{
                        VC.delegate = controller
                    }
                }
            }
        }
        if settings.type == .kReservation{
            if let topVC = UIApplication.topViewController(){
                if let VC = ReservationDetailVC.instance(model: settings.reservationModel){
                    if let controller = VC.childViewController as? ReservationDetailVC{
                        controller.delegate = (topVC as! ReservationListVC)
                    }
                    topVC.present(VC, animated: true, completion: nil)
                }
            }
        }
    }
    
    //MARK: - UIAlertController
    func showAlertView(parentVC:UIViewController, withAlertTile title:String?, withAlertMsg msg:String, withActionsTitle btnTitles:[String], withHandler handler:@escaping ((Int) -> Void )){
        if msg == "" || isPresented{
            return
        }
        self.isPresented = true
        self.handler = handler
        var setting = AlertSetting()
        setting.alertMessage = msg
        setting.alertTitle = title ?? ""
        if btnTitles.count == 1{
            setting.confirmTitle = btnTitles.first ?? ""
            setting.cancelTitle = ""
        }else{
            setting.confirmTitle = btnTitles.last ?? ""
            setting.cancelTitle = btnTitles.first ?? ""
        }
        if let VC = AlertViewController.showPopup(parentVC: getParentView(parentVC: parentVC), setting: setting){
            VC.delegate = self
        }
    }
    
    func getParentView(parentVC:UIViewController)->UIViewController{
        if let VC = parentVC.parent as? UINavigationController{
            return VC
        }else if let VC = parentVC.parent{
            return self.getParentView(parentVC: VC)
        }else{
            return parentVC
        }
    }
    
    //MARK: Validation
    func isValidRegEx(_ testStr: String, _ regex: RegEx) -> Bool {
        let stringTest = NSPredicate(format:"SELF MATCHES %@", regex.rawValue)
        let result = stringTest.evaluate(with: testStr)
        return result
    }
    
    func alertButtonAction(index: Int, data: alertViewData, type: AlertType) {
        self.isPresented = false
        handler(index)
    }
}
